<?php

namespace App\Console\Commands;

use App\Models\Student;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class SeedDummyStudents extends Command
{
    protected $signature = 'students:seed-dummy
                            {count=10000 : Number of dummy students to create}
                            {--chunk=500 : Insert chunk size}';

    protected $description = 'Add dummy students for speed testing, divided equally across all categories (Active, Pending, Alumna, Not Eligible, Information Pending)';

    public function handle(): int
    {
        $count = (int) $this->argument('count');
        $chunkSize = (int) $this->option('chunk');

        if ($count < 1 || $chunkSize < 1) {
            $this->error('Count and chunk must be positive.');
            return self::FAILURE;
        }

        $perCategory = (int) floor($count / 5);
        if ($perCategory < 1) {
            $this->error('Count must be at least 5 to divide into 5 categories.');
            return self::FAILURE;
        }

        $categories = [
            ['status' => Student::STATUS_ACTIVE, 'form_status' => Student::FORM_STATUS_ACTIVE, 'label' => 'Active'],
            ['status' => Student::STATUS_PENDING, 'form_status' => Student::FORM_STATUS_ACTIVE, 'label' => 'Pending'],
            ['status' => Student::STATUS_LEFT, 'form_status' => Student::FORM_STATUS_ACTIVE, 'label' => 'Alumna'],
            ['status' => Student::STATUS_CANCELLED, 'form_status' => Student::FORM_STATUS_ACTIVE, 'label' => 'Student Not Eligible'],
            ['status' => Student::STATUS_PENDING, 'form_status' => null, 'label' => 'Information Pending'],
        ];

        // Information Pending: rotate form_status among pending types
        $infoPendingFormStatuses = [
            Student::FORM_STATUS_PENDING,
            Student::FORM_STATUS_DOCUMENTS_UPLOADED,
            Student::FORM_STATUS_BED_ALLOTTED,
            Student::FORM_STATUS_DEPOSIT_PAID,
        ];

        $this->info("Creating {$count} dummy students ({$perCategory} per category)...");
        $bar = $this->output->createProgressBar($count);
        $bar->start();

        $now = now();
        $globalIndex = 0;

        foreach ($categories as $cat) {
            $rows = [];
            for ($i = 0; $i < $perCategory; $i++) {
                $globalIndex++;
                $formStatus = $cat['form_status']
                    ?? $infoPendingFormStatuses[$i % 4];
                $rows[] = [
                    'first_name' => 'DummyFirst' . $globalIndex,
                    'middle_name' => null,
                    'last_name' => 'Student' . $globalIndex,
                    'email' => 'dummy' . $globalIndex . '@test.local',
                    'date_of_birth' => null,
                    'permanent_locality' => 'Test Locality',
                    'permanent_taluka' => null,
                    'permanent_district' => 'Test District',
                    'permanent_state' => 'Test State',
                    'student_mobile' => '9' . str_pad((string) ($globalIndex % 100000000), 9, '0', STR_PAD_LEFT),
                    'parent_mobile' => null,
                    'aadhar_number' => null,
                    'standard_year' => null,
                    'stream' => null,
                    'college_name' => null,
                    'college_in_time' => null,
                    'college_out_time' => null,
                    'reference_name' => null,
                    'reference_relation' => null,
                    'reference_contact' => null,
                    'reference_by_student_id' => null,
                    'terms_accepted_at' => $now,
                    'form_status' => $formStatus,
                    'status' => $cat['status'],
                    'deposit_status' => false,
                    'floor_id' => null,
                    'room_id' => null,
                    'bed_id' => null,
                    'room_bed_identifier' => null,
                    'reward_points' => 0,
                    'created_at' => $now,
                    'updated_at' => $now,
                ];
            }

            foreach (array_chunk($rows, $chunkSize) as $chunk) {
                DB::table('students')->insert($chunk);
                $bar->advance(count($chunk));
            }
        }

        $bar->finish();
        $this->newLine(2);
        $this->info("Done. Created {$count} dummy students.");
        $this->table(
            ['Category', 'Count'],
            [
                ['Active', $perCategory],
                ['Pending', $perCategory],
                ['Alumna', $perCategory],
                ['Student Not Eligible', $perCategory],
                ['Information Pending', $perCategory],
            ]
        );
        return self::SUCCESS;
    }
}
