<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Auth;

class MenuHelper
{
    /**
     * Permission mapping for menu items
     * Maps custom page slugs to their required permissions
     *
     * @var array
     */
    protected static $permissionMap = [
        // Content Management
        'custom_pages' => 'manage_custom_pages',
        'email_templates' => 'manage_email_templates',
        'menu_management' => 'manage_menus',
        
        // User Management
        'users' => 'view_users',
        'create_user' => 'create_users',
        'user_roles' => 'manage_roles',
        'permissions' => 'manage_permissions',
        
        // Ecommerce
        'store' => 'view_store',
        'products' => 'manage_products',
        'invoices' => 'view_invoices',
        'orders' => 'view_orders',
        'payments' => 'view_payments',
        
        // Settings
        'api_configuration' => 'manage_api_settings',
        'website_configuration' => 'manage_website_settings',
        'general_settings' => 'manage_general_settings',
        'security_settings' => 'manage_security_settings',
        
        // Events & Academy
        'events' => 'view_events',
        'create_event' => 'create_events',
        'courses' => 'manage_courses',
        'lessons' => 'manage_lessons',
        'quizzes' => 'view_quizzes',
        'quiz_results' => 'view_quiz_results',
        
        // Reports & Analytics
        'reports' => 'view_reports',
        'analytics' => 'view_analytics',
        'audit_logs' => 'view_audit_logs',
        
        // Communication
        'notifications' => 'manage_notifications',
        'messages' => 'view_messages',
        'announcements' => 'manage_announcements',
    ];

    /**
     * Instructor allowlist - slugs that instructors can access
     *
     * @var array
     */
    protected static $instructorAllowlist = [
        'courses',
        'lessons',
        'quizzes',
        'quiz_results',
        'dashboard',
        'profile',
    ];

    /**
     * Check if authenticated user can access menu item
     *
     * @param array|object $menuItem
     * @return bool
     */
    public static function canAccessMenuItem($menuItem): bool
    {
        $user = Auth::user();

        // If no user is authenticated, deny access
        if (!$user) {
            return false;
        }

        // Convert object to array if needed
        if (is_object($menuItem)) {
            $menuItem = (array) $menuItem;
        }

        // Get the slug/identifier for this menu item
        $slug = $menuItem['custom_page'] ?? $menuItem['slug'] ?? null;

        // If no slug, check link_type
        if (!$slug && isset($menuItem['link_type'])) {
            // Placeholder items are visible to all authenticated users
            if ($menuItem['link_type'] === 'placeholder') {
                return true;
            }
            
            // Link URLs are visible to all authenticated users
            if ($menuItem['link_type'] === 'link_url') {
                return true;
            }
        }

        // Super admin can access everything
        if (isset($user->role) && $user->role === 'super_admin') {
            return true;
        }

        // Admin can access most things (you can customize this)
        if (isset($user->role) && $user->role === 'admin') {
            return true;
        }

        // Instructor role-based access
        if (isset($user->role) && $user->role === 'instructor') {
            // Check if slug is in instructor allowlist
            if ($slug && in_array($slug, self::$instructorAllowlist)) {
                return true;
            }
            // Instructors don't have access to items not in allowlist
            return false;
        }

        // For other roles, check permission mapping
        if ($slug && isset(self::$permissionMap[$slug])) {
            $requiredPermission = self::$permissionMap[$slug];
            
            // Check if user has the required permission
            // This assumes you have a permissions system in place
            if (method_exists($user, 'hasPermission')) {
                return $user->hasPermission($requiredPermission);
            }
            
            // Fallback: check if user has permissions property
            if (isset($user->permissions) && is_array($user->permissions)) {
                return in_array($requiredPermission, $user->permissions);
            }
        }

        // Default deny for undefined items or users without proper permissions
        return false;
    }

    /**
     * Recursively filter menu array based on user permissions
     *
     * @param array $menu
     * @return array
     */
    public static function filterMenuByPermissions(array $menu): array
    {
        $filteredMenu = [];

        foreach ($menu as $menuItem) {
            // Convert to array if it's an object
            $item = is_object($menuItem) ? (array) $menuItem : $menuItem;

            // Check if user can access this menu item
            if (self::canAccessMenuItem($item)) {
                $filteredItem = $item;

                // Recursively filter submenus (level 1)
                if (isset($item['submenus']) && is_array($item['submenus']) && !empty($item['submenus'])) {
                    $filteredSubmenus = self::filterSubmenusByPermissions($item['submenus']);
                    
                    // Only include parent if it has accessible submenus or is itself a link
                    if (!empty($filteredSubmenus)) {
                        $filteredItem['submenus'] = $filteredSubmenus;
                        $filteredMenu[] = $filteredItem;
                    } elseif (
                        (isset($item['link_type']) && $item['link_type'] !== 'placeholder') ||
                        !empty($item['link_url']) ||
                        !empty($item['custom_page'])
                    ) {
                        // Parent has a link itself, so include it even without submenus
                        $filteredItem['submenus'] = [];
                        $filteredMenu[] = $filteredItem;
                    }
                } else {
                    // No submenus, just add the item
                    $filteredMenu[] = $filteredItem;
                }
            }
        }

        return $filteredMenu;
    }

    /**
     * Recursively filter nested submenu items (unlimited levels)
     *
     * @param array $submenus
     * @return array
     */
    protected static function filterSubmenusByPermissions(array $submenus): array
    {
        $filteredSubmenus = [];

        foreach ($submenus as $submenuItem) {
            // Convert to array if it's an object
            $item = is_object($submenuItem) ? (array) $submenuItem : $submenuItem;

            // Check if user can access this submenu item
            if (self::canAccessMenuItem($item)) {
                $filteredItem = $item;

                // Recursively filter nested submenus
                if (isset($item['submenus']) && is_array($item['submenus']) && !empty($item['submenus'])) {
                    $filteredNestedSubmenus = self::filterSubmenusByPermissions($item['submenus']);
                    
                    // Only include submenu if it has accessible nested items or is itself a link
                    if (!empty($filteredNestedSubmenus)) {
                        $filteredItem['submenus'] = $filteredNestedSubmenus;
                        $filteredSubmenus[] = $filteredItem;
                    } elseif (
                        (isset($item['link_type']) && $item['link_type'] !== 'placeholder') ||
                        !empty($item['link_url']) ||
                        !empty($item['custom_page'])
                    ) {
                        // Submenu has a link itself, so include it even without nested items
                        $filteredItem['submenus'] = [];
                        $filteredSubmenus[] = $filteredItem;
                    }
                } else {
                    // No nested submenus, just add the item
                    $filteredSubmenus[] = $filteredItem;
                }
            }
        }

        return $filteredSubmenus;
    }

    /**
     * Add a custom permission mapping
     *
     * @param string $slug
     * @param string $permission
     * @return void
     */
    public static function addPermissionMapping(string $slug, string $permission): void
    {
        self::$permissionMap[$slug] = $permission;
    }

    /**
     * Add slug to instructor allowlist
     *
     * @param string $slug
     * @return void
     */
    public static function addToInstructorAllowlist(string $slug): void
    {
        if (!in_array($slug, self::$instructorAllowlist)) {
            self::$instructorAllowlist[] = $slug;
        }
    }

    /**
     * Get all permission mappings
     *
     * @return array
     */
    public static function getPermissionMap(): array
    {
        return self::$permissionMap;
    }

    /**
     * Get instructor allowlist
     *
     * @return array
     */
    public static function getInstructorAllowlist(): array
    {
        return self::$instructorAllowlist;
    }
}
