<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AgreementHistory;
use App\Models\Student;
use App\Models\StudentRemark;
use Carbon\Carbon;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Illuminate\View\View;

class AgreementController extends Controller
{
    /**
     * Agreement list with expiry filters; optional student_id to focus one student.
     */
    public function index(Request $request): View
    {
        $studentId = $request->query('student_id') ? (int) $request->query('student_id') : null;
        $filter = $request->query('filter', 'all'); // all | expiring_today | expiring_week | expiring_30
        $today = Carbon::today();

        $query = Student::query()
            ->with(['bed', 'room', 'floor', 'documents', 'agreementHistories', 'remarks'])
            ->where('form_status', Student::FORM_STATUS_ACTIVE)
            ->whereHas('agreementHistories');

        if ($studentId) {
            $query->where('id', $studentId);
        }

        $latestToDateSub = AgreementHistory::query()
            ->selectRaw('student_id, MAX(to_date) as latest_to_date')
            ->groupBy('student_id');

        if ($filter === 'expiring_today') {
            $query->joinSub($latestToDateSub, 'latest_ag', function ($join) {
                $join->on('students.id', '=', 'latest_ag.student_id');
            })->whereDate('latest_ag.latest_to_date', $today)->select('students.*');
        } elseif ($filter === 'expiring_week') {
            $query->joinSub($latestToDateSub, 'latest_ag', function ($join) {
                $join->on('students.id', '=', 'latest_ag.student_id');
            })->whereBetween('latest_ag.latest_to_date', [$today->copy()->addDay(), $today->copy()->addDays(7)])->select('students.*');
        } elseif ($filter === 'expiring_30') {
            $query->joinSub($latestToDateSub, 'latest_ag', function ($join) {
                $join->on('students.id', '=', 'latest_ag.student_id');
            })->whereBetween('latest_ag.latest_to_date', [$today->copy()->addDays(8), $today->copy()->addDays(30)])->select('students.*');
        }

        $students = $query->orderBy('first_name')->orderBy('last_name')->get();

        return view('admin.agreements.index', [
            'students' => $students,
            'filter' => $filter,
            'studentId' => $studentId,
        ]);
    }

    /**
     * Print formal agreement for a student.
     */
    public function view(Student $student): View
    {
        $student->load(['bed', 'room', 'floor', 'documents', 'agreementHistories']);
        $firstAgreement = $student->agreementHistories->sortBy('from_date')->first();
        $latestAgreement = $student->agreementHistories->sortByDesc('to_date')->first();
        $admissionDate = $firstAgreement?->from_date;
        $leavingDate = $latestAgreement?->to_date;

        $pdfDownloadUrl = route('admin.agreements.pdf', $student);

        return view('admin.agreements.view', [
            'student' => $student,
            'admissionDate' => $admissionDate,
            'leavingDate' => $leavingDate,
            'pdfDownloadUrl' => $pdfDownloadUrl,
            'emailType' => 'agreement',
            'emailId' => $student->id,
        ]);
    }

    /**
     * Form to renew agreement (increase to_date).
     */
    public function renew(Student $student): View|RedirectResponse
    {
        $student->load('agreementHistories');
        $latest = $student->agreementHistories->sortByDesc('to_date')->first();
        if (! $latest) {
            return redirect()->route('admin.agreements.index')->with('error', 'No agreement record found for this student.');
        }

        return view('admin.agreements.renew', [
            'student' => $student,
            'currentToDate' => $latest->to_date,
        ]);
    }

    /**
     * Store agreement renewal: new agreement_history row + optional remark.
     */
    public function storeRenewal(Request $request, Student $student): RedirectResponse
    {
        $validated = $request->validate([
            'increase_to_date' => ['required', 'date', 'after_or_equal:today'],
            'remark' => ['nullable', 'string', 'max:1000'],
        ]);

        $student->load('agreementHistories');
        $latest = $student->agreementHistories->sortByDesc('to_date')->first();
        if (! $latest) {
            return redirect()->route('admin.agreements.index')->with('error', 'No agreement record found.');
        }

        AgreementHistory::query()->create([
            'student_id' => $student->id,
            'from_date' => $latest->from_date,
            'to_date' => $validated['increase_to_date'],
        ]);

        if (! empty(trim($validated['remark'] ?? ''))) {
            StudentRemark::query()->create([
                'student_id' => $student->id,
                'remark' => trim($validated['remark']),
                'remarked_at' => now()->toDateString(),
            ]);
        }

        return redirect()
            ->route('admin.agreements.index', ['student_id' => $student->id])
            ->with('success', 'Agreement renewed successfully. New leaving date: ' . Carbon::parse($validated['increase_to_date'])->format('d/m/Y'));
    }

    /**
     * Store a remark for a student (from agreement index or student show).
     */
    public function storeRemark(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'student_id' => ['required', 'integer', 'exists:students,id'],
            'remark' => ['required', 'string', 'max:1000'],
            'remarked_at' => ['required', 'date'],
        ]);

        StudentRemark::query()->create([
            'student_id' => $validated['student_id'],
            'remark' => $validated['remark'],
            'remarked_at' => $validated['remarked_at'],
        ]);

        $back = $request->query('back', route('admin.agreements.index', ['student_id' => $validated['student_id']]));

        return redirect($back)->with('success', 'Remark added.');
    }
}
