<?php

namespace App\Http\Controllers\Admin\Auth;

use App\Http\Controllers\Controller;
use App\Services\OtpService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Validation\ValidationException;

class VerifyOtpController extends Controller
{
    public function __construct(
        protected OtpService $otpService
    ) {}

    public function showVerifyForm(Request $request)
    {
        if (!Auth::guard('admin')->check()) {
            return redirect()->route('admin.login');
        }
        $user = Auth::guard('admin')->user();
        $maskedEmail = OtpService::maskEmail($user->email);
        $devOtp = app()->environment('local') ? config('otp.static_otp', '123456') : null;
        return view('auth.admin.verify-otp', compact('maskedEmail', 'devOtp'));
    }

    public function verify(Request $request)
    {
        $request->validate([
            'otp' => ['required', 'string', 'size:6'],
        ]);

        $key = 'otp-verify-admin-' . (Auth::guard('admin')->id() ?? $request->ip());
        if (RateLimiter::tooManyAttempts($key, 5)) {
            $seconds = RateLimiter::availableIn($key);
            throw ValidationException::withMessages([
                'otp' => ['Too many attempts. Please try again in ' . $seconds . ' seconds.'],
            ]);
        }

        $user = Auth::guard('admin')->user();
        if (!$user) {
            return redirect()->route('admin.login');
        }

        if (!$this->otpService->verify($user, 'admin', $request->otp)) {
            RateLimiter::hit($key);
            throw ValidationException::withMessages([
                'otp' => [__('The verification code is invalid or has expired.')],
            ]);
        }

        RateLimiter::clear($key);
        session(['otp_verified_admin' => true]);

        return redirect()->intended(route('admin.dashboard'));
    }

    public function resend(Request $request)
    {
        $key = 'otp-resend-admin-' . (Auth::guard('admin')->id() ?? $request->ip());
        if (RateLimiter::tooManyAttempts($key, 3)) {
            $seconds = RateLimiter::availableIn($key);
            return back()->with('error', __('Please wait :seconds seconds before requesting a new code.', ['seconds' => $seconds]));
        }

        $user = Auth::guard('admin')->user();
        if (!$user) {
            return redirect()->route('admin.login');
        }

        $this->otpService->generate($user, 'admin');
        RateLimiter::hit($key);

        return back()->with('status', __('A new verification code has been sent to your email.'));
    }
}
