<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BankEntry;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class BankDepositController extends Controller
{
    /**
     * Bank transactions list: deposit to bank / withdraw from bank. Filters + total.
     */
    public function index(Request $request): View
    {
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date');
        $type = $request->query('type');
        $paymentMode = $request->query('payment_mode');

        $query = BankEntry::query()
            ->orderByDesc('entry_date')
            ->orderByDesc('id');

        if ($fromDate) {
            $query->whereDate('entry_date', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('entry_date', '<=', $toDate);
        }
        if ($type !== null && $type !== '') {
            $query->where('type', $type);
        }
        if ($paymentMode !== null && $paymentMode !== '') {
            $query->where('payment_mode', $paymentMode);
        }

        $entries = $query->get();
        $total = $entries->sum('amount');

        $paymentModes = config('student_registration.payment_modes', []);

        return view('admin.bank-deposits.index', [
            'entries' => $entries,
            'total' => $total,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'type' => $type,
            'paymentMode' => $paymentMode,
            'paymentModes' => $paymentModes,
        ]);
    }

    /**
     * Store a bank entry (deposit to bank or withdraw from bank).
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'entry_date' => ['required', 'date'],
            'amount' => ['required', 'numeric', 'min:0.01'],
            'type' => ['required', Rule::in([BankEntry::TYPE_DEPOSIT_TO_BANK, BankEntry::TYPE_WITHDRAW_FROM_BANK])],
            'payment_mode' => ['nullable', 'string', 'max:50'],
            'person_name' => ['nullable', 'string', 'max:255'],
            'transaction_id' => ['nullable', 'string', 'max:255'],
        ]);

        BankEntry::query()->create([
            'entry_date' => $validated['entry_date'],
            'amount' => $validated['amount'],
            'type' => $validated['type'],
            'payment_mode' => $validated['payment_mode'] ?? null,
            'person_name' => $validated['person_name'] ?? null,
            'transaction_id' => $validated['transaction_id'] ?? null,
        ]);

        $label = $validated['type'] === BankEntry::TYPE_DEPOSIT_TO_BANK ? 'Deposit to Bank' : 'Withdraw from Bank';
        return redirect()
            ->route('admin.bank-deposits.index')
            ->with('success', $label . ' recorded successfully.');
    }
}
