<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AgreementHistory;
use App\Models\Deposit;
use App\Models\DepositReturn;
use App\Models\Student;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class DepositController extends Controller
{
    /**
     * Deposit Details page: tabs for Received, Pending, Return, Pending Release Student.
     */
    public function index(Request $request): View
    {
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date');

        // Deposit Received: students with deposit paid and NOT yet returned (active residents only)
        $depositsReceivedQuery = Deposit::query()
            ->with(['student.documents'])
            ->whereHas('student', fn ($q) => $q->where('deposit_status', true)->whereDoesntHave('depositReturns'));
        if ($fromDate) {
            $depositsReceivedQuery->whereDate('actual_deposit_date', '>=', $fromDate);
        }
        if ($toDate) {
            $depositsReceivedQuery->whereDate('actual_deposit_date', '<=', $toDate);
        }
        $depositsReceived = $depositsReceivedQuery->orderByDesc('actual_deposit_date')->get();

        // Deposit Pending: bed allotted but no deposit / deposit_status false
        $depositPendingQuery = Student::query()
            ->with('documents')
            ->where('form_status', Student::FORM_STATUS_BED_ALLOTTED)
            ->where(function ($q) {
                $q->where('deposit_status', false)->orWhereDoesntHave('deposits');
            });
        $depositPending = $depositPendingQuery->orderBy('first_name')->get();

        // Deposit Return: archive of refunded deposits
        $depositReturnQuery = DepositReturn::query()
            ->with(['student.documents'])
            ->orderByDesc('return_date');
        if ($fromDate) {
            $depositReturnQuery->whereDate('return_date', '>=', $fromDate);
        }
        if ($toDate) {
            $depositReturnQuery->whereDate('return_date', '<=', $toDate);
        }
        $depositReturn = $depositReturnQuery->get();
        $depositReturnTotal = $depositReturn->sum('amount_returned');

        // Deposit Pending Release Student: left but deposit not yet returned
        $depositPendingReleaseQuery = Student::query()
            ->with('documents')
            ->where('status', Student::STATUS_LEFT)
            ->where('deposit_status', true)
            ->whereDoesntHave('depositReturns');
        $depositPendingRelease = $depositPendingReleaseQuery->orderByDesc('updated_at')->get();

        // Stats for dashboard cards
        $stats = [
            'received_count' => $depositsReceived->count(),
            'received_total' => $depositsReceived->sum('amount'),
            'pending_count' => $depositPending->count(),
            'return_count' => $depositReturn->count(),
            'return_total' => $depositReturnTotal,
            'pending_release_count' => $depositPendingRelease->count(),
        ];

        return view('admin.deposits.index', [
            'depositsReceived' => $depositsReceived,
            'depositPending' => $depositPending,
            'depositReturn' => $depositReturn,
            'depositReturnTotal' => $depositReturnTotal,
            'depositPendingRelease' => $depositPendingRelease,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'stats' => $stats,
        ]);
    }

    /**
     * Show the Add Deposit form (separate page, not student registration).
     * When student_id is in the URL (from Pending list), that student is shown as read-only.
     * When opened from top "Add Deposit" button, the student dropdown is shown.
     */
    public function create(Request $request): View
    {
        $paymentModes = config('student_registration.payment_modes', []);
        $students = Student::query()
            ->where('form_status', Student::FORM_STATUS_BED_ALLOTTED)
            ->where(function ($q) {
                $q->where('deposit_status', false)->orWhereDoesntHave('deposits');
            })
            ->orderBy('first_name')
            ->orderBy('last_name')
            ->get(['id', 'first_name', 'middle_name', 'last_name', 'student_mobile']);

        $selectedStudent = null;
        if ($request->filled('student_id')) {
            $id = (int) $request->query('student_id');
            $selectedStudent = $students->firstWhere('id', $id);
        }

        return view('admin.deposits.add', [
            'paymentModes' => $paymentModes,
            'students' => $students,
            'selectedStudent' => $selectedStudent,
        ]);
    }

    /**
     * Store a new deposit for the selected student.
     */
    public function store(Request $request): RedirectResponse
    {
        $paymentModes = array_keys(config('student_registration.payment_modes', []));
        $validated = $request->validate([
            'student_id' => ['required', 'integer', 'exists:students,id'],
            'from_date' => ['required', 'date'],
            'to_date' => ['required', 'date', 'after_or_equal:from_date'],
            'actual_deposit_date' => ['required', 'date'],
            'amount' => ['required', 'numeric', 'min:0'],
            'payment_mode' => ['required', Rule::in($paymentModes)],
            'transaction_id' => ['nullable', 'string', 'max:255'],
        ]);

        $student = Student::query()->findOrFail($validated['student_id']);
        $requiredAmount = hostel_deposit_amount();
        $amount = (float) $validated['amount'];
        $balanceAmount = $amount < $requiredAmount ? $requiredAmount - $amount : 0;
        $advanceAmount = $amount > $requiredAmount ? $amount - $requiredAmount : 0;

        // Ensure student is eligible (bed allotted, no deposit yet)
        if ($student->form_status !== Student::FORM_STATUS_BED_ALLOTTED) {
            return back()->withInput()->withErrors(['student_id' => 'This student is not eligible for deposit (bed must be allotted first).']);
        }
        if ($student->deposit_status || $student->deposits()->exists()) {
            return back()->withInput()->withErrors(['student_id' => 'This student already has a deposit recorded.']);
        }

        DB::transaction(function () use ($student, $validated, $balanceAmount, $advanceAmount) {
            Deposit::query()->create([
                'student_id' => $student->id,
                'from_date' => $validated['from_date'],
                'to_date' => $validated['to_date'],
                'actual_deposit_date' => $validated['actual_deposit_date'],
                'amount' => $validated['amount'],
                'payment_mode' => $validated['payment_mode'],
                'transaction_id' => $validated['transaction_id'] ?? null,
                'balance_amount' => $balanceAmount,
                'advance_amount' => $advanceAmount,
            ]);
            AgreementHistory::query()->create([
                'student_id' => $student->id,
                'from_date' => $validated['from_date'],
                'to_date' => $validated['to_date'],
            ]);
            $student->update([
                'form_status' => Student::FORM_STATUS_DEPOSIT_PAID,
                'deposit_status' => true,
            ]);
        });

        return redirect()->route('admin.deposits.create')->with('success', 'Deposit added successfully for ' . $student->full_name . '.');
    }

    /**
     * Show the Return Deposit form for a student (when they are leaving).
     */
    public function returnForm(Student $student): View|RedirectResponse
    {
        if (! $student->deposit_status) {
            return redirect()->route('admin.deposits.create')->with('error', 'Student has no deposit to return.');
        }
        if ($student->depositReturns()->exists()) {
            return redirect()->route('admin.deposits.create')->with('error', 'Deposit has already been returned for this student.');
        }
        if (! in_array($student->status, [Student::STATUS_ACTIVE, Student::STATUS_LEFT], true)) {
            return redirect()->route('admin.deposits.create')->with('error', 'Student is not eligible for deposit return.');
        }

        $paymentModes = config('student_registration.payment_modes', []);
        $latestDeposit = $student->deposits()->orderByDesc('actual_deposit_date')->first();

        return view('admin.deposits.return', [
            'student' => $student,
            'paymentModes' => $paymentModes,
            'latestDeposit' => $latestDeposit,
        ]);
    }

    /**
     * Store a deposit return (refund) and mark student as left.
     */
    public function storeReturn(Request $request): RedirectResponse
    {
        $paymentModes = array_keys(config('student_registration.payment_modes', []));
        $validated = $request->validate([
            'student_id' => ['required', 'integer', 'exists:students,id'],
            'amount_returned' => ['required', 'numeric', 'min:0'],
            'return_date' => ['required', 'date'],
            'payment_mode' => ['required', Rule::in($paymentModes)],
            'transaction_id' => ['nullable', 'string', 'max:255'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);

        $student = Student::query()->findOrFail($validated['student_id']);

        if (! $student->deposit_status) {
            return back()->withInput()->withErrors(['student_id' => 'Student has no deposit to return.']);
        }
        if ($student->depositReturns()->exists()) {
            return back()->withInput()->withErrors(['student_id' => 'Deposit has already been returned for this student.']);
        }

        DB::transaction(function () use ($student, $validated) {
            DepositReturn::query()->create([
                'student_id' => $student->id,
                'amount_returned' => $validated['amount_returned'],
                'return_date' => $validated['return_date'],
                'payment_mode' => $validated['payment_mode'],
                'transaction_id' => $validated['transaction_id'] ?? null,
                'notes' => $validated['notes'] ?? null,
            ]);
            // Mark student as left (alumna) and deposit no longer held
            $student->update([
                'status' => Student::STATUS_LEFT,
                'deposit_status' => false,
            ]);
        });

        return redirect()->route('admin.deposits.create')->with('success', 'Deposit returned successfully. ' . $student->full_name . ' is now in Alumna (left hostel) and appears under Deposit Return.');
    }

    /**
     * Print single deposit receipt (for Receipt Book re-generation).
     */
    public function receipt(Request $request, Deposit $deposit): View|JsonResponse
    {
        $deposit->load(['student.bed', 'student.room', 'student.floor']);
        $amountInWords = $this->amountToWords((float) $deposit->amount);
        
        // If AJAX request, return JSON with HTML content for modal
        if ($request->wantsJson() || $request->ajax()) {
            $html = view('admin.receipts._deposit_content', [
                'deposit' => $deposit,
                'amountInWords' => $amountInWords,
            ])->render();
            $title = 'Deposit Receipt - D-' . $deposit->id;

            return response()->json([
                'success' => true,
                'html' => $html,
                'title' => $title,
                'pdfDownloadUrl' => route('admin.receipts.deposit.pdf', $deposit),
                'receiptNo' => 'D-' . $deposit->id,
                'emailType' => 'deposit',
                'emailId' => $deposit->id,
            ]);
        }
        
        $pdfDownloadUrl = route('admin.receipts.deposit.pdf', $deposit);

        return view('admin.deposits.receipt', [
            'deposit' => $deposit,
            'amountInWords' => $amountInWords,
            'pdfDownloadUrl' => $pdfDownloadUrl,
            'emailType' => 'deposit',
            'emailId' => $deposit->id,
        ]);
    }

    /**
     * Print single deposit return receipt (for Receipt Book re-generation).
     */
    public function returnReceipt(Request $request, DepositReturn $depositReturn): View|JsonResponse
    {
        $depositReturn->load(['student.bed', 'student.room', 'student.floor']);
        $amountInWords = $this->amountToWords((float) $depositReturn->amount_returned);
        
        // If AJAX request, return JSON with HTML content for modal
        if ($request->wantsJson() || $request->ajax()) {
            $html = view('admin.receipts._deposit_return_content', [
                'depositReturn' => $depositReturn,
                'amountInWords' => $amountInWords,
            ])->render();
            $title = 'Deposit Return Receipt - DR-' . $depositReturn->id;

            return response()->json([
                'success' => true,
                'html' => $html,
                'title' => $title,
                'pdfDownloadUrl' => route('admin.receipts.deposit-return.pdf', $depositReturn),
                'receiptNo' => 'DR-' . $depositReturn->id,
                'emailType' => 'deposit_return',
                'emailId' => $depositReturn->id,
            ]);
        }
        
        $pdfDownloadUrl = route('admin.receipts.deposit-return.pdf', $depositReturn);

        return view('admin.deposits.return-receipt', [
            'depositReturn' => $depositReturn,
            'amountInWords' => $amountInWords,
            'pdfDownloadUrl' => $pdfDownloadUrl,
            'emailType' => 'deposit_return',
            'emailId' => $depositReturn->id,
        ]);
    }

    private function amountToWords(float $amount): string
    {
        $rupees = (int) floor($amount);
        $paise = (int) round(($amount - $rupees) * 100);
        $words = $this->numberToWords($rupees);
        $out = $words ? ucfirst($words) . ' Rupees' : 'Zero Rupees';
        if ($paise > 0) {
            $out .= ' and ' . ucfirst($this->numberToWords($paise)) . ' Paise';
        }
        return $out . ' only';
    }

    private function numberToWords(int $n): string
    {
        if ($n === 0) {
            return '';
        }
        $ones = ['', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine', 'Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen', 'Seventeen', 'Eighteen', 'Nineteen'];
        $tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];
        if ($n < 20) {
            return $ones[$n];
        }
        if ($n < 100) {
            return trim($tens[(int) floor($n / 10)] . ' ' . $ones[$n % 10]);
        }
        if ($n < 1000) {
            $h = (int) floor($n / 100);
            $r = $n % 100;
            return trim($ones[$h] . ' Hundred ' . $this->numberToWords($r));
        }
        if ($n < 100000) {
            $th = (int) floor($n / 1000);
            $r = $n % 1000;
            return trim($this->numberToWords($th) . ' Thousand ' . $this->numberToWords($r));
        }
        if ($n < 10000000) {
            $l = (int) floor($n / 100000);
            $r = $n % 100000;
            return trim($this->numberToWords($l) . ' Lakh ' . $this->numberToWords($r));
        }
        $c = (int) floor($n / 10000000);
        $r = $n % 10000000;
        return trim($this->numberToWords($c) . ' Crore ' . $this->numberToWords($r));
    }
}
