<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Menu;
use App\Models\CustomPageModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class MenuManagementController extends Controller
{
    /**
     * Display menu management page (Legacy version - Active menus only)
     */
    public function menuManagement()
    {
        try {
            $menus = [];
            Menu::where('is_active', 1)
                ->orderBy('order')
                ->chunk(100, function ($chunk) use (&$menus) {
                    foreach ($chunk as $menu) {
                        $menuArray = $menu->toArray();
                        $menuArray['submenus'] = !empty($menu->submenus) 
                            ? json_decode($menu->submenus, true) 
                            : [];
                        $menus[] = $menuArray;
                    }
                });

            $customPages = CustomPageModel::where('status', 1)->get();

            return view('content.admin.menu_management.menu_management', compact('menus', 'customPages'));
        } catch (\Exception $e) {
            Log::error('Menu Management Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to load menu management page.');
        }
    }

    /**
     * Display menu management page (Improved version - All menus)
     */
    public function menuManagementImproved()
    {
        try {
            $menus = [];
            Menu::orderBy('order')
                ->chunk(100, function ($chunk) use (&$menus) {
                    foreach ($chunk as $menu) {
                        $menuArray = $menu->toArray();
                        $menuArray['submenus'] = !empty($menu->submenus) 
                            ? json_decode($menu->submenus, true) 
                            : [];
                        $menus[] = $menuArray;
                    }
                });

            $customPages = CustomPageModel::where('status', 1)->get();

            return view('content.admin.menu_management.menu_management_improved', compact('menus', 'customPages'));
        } catch (\Exception $e) {
            Log::error('Menu Management Improved Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to load menu management page.');
        }
    }

    /**
     * Display menu management page (V2 - Current/Default version)
     */
    public function menuManagementV2()
    {
        try {
            $menus = Menu::orderBy('order')->get();
            $customPages = CustomPageModel::where('status', 1)->get();

            return view('content.admin.menu_management.menu_management_v2', compact('menus', 'customPages'));
        } catch (\Exception $e) {
            Log::error('Menu Management V2 Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to load menu management page.');
        }
    }

    /**
     * Store a new menu
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required|string|max:255|unique:content_menus,title',
                'title_text' => 'required|string|max:255',
                'main_link_type' => 'required|in:custom_page,link_url,placeholder',
                'main_order' => 'required|integer|min:1',
            ]);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Validation failed. Please check the form.');
            }

            // Handle link type logic
            $customPage = null;
            $linkUrl = null;

            if ($request->main_link_type === 'custom_page') {
                $customPage = $request->input('main_custom_page');
            } elseif ($request->main_link_type === 'link_url') {
                $linkUrl = $request->input('main_link_url');
            }

            // Process submenus if provided
            $submenus = [];
            if ($request->has('submenus') && is_array($request->submenus)) {
                $submenus = $this->processSubmenus($request->submenus);
            }

            Menu::create([
                'title' => $request->title,
                'title_text' => $request->title_text,
                'link_type' => $request->main_link_type,
                'custom_page' => $customPage,
                'link_url' => $linkUrl,
                'order' => $request->main_order,
                'submenus' => json_encode($submenus),
                'is_active' => $request->has('is_active') ? 1 : 0,
            ]);

            return redirect()->route('admin.content-menu_management')->with('success', 'Menu created successfully!');
        } catch (\Exception $e) {
            Log::error('Menu Store Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to create menu.')->withInput();
        }
    }

    /**
     * Quick store menu (AJAX)
     */
    public function quickStore(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required|string|max:255|unique:content_menus,title',
                'title_text' => 'required|string|max:255',
                'link_type' => 'required|in:custom_page,link_url,placeholder',
                'order' => 'required|integer|min:1',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $customPage = null;
            $linkUrl = null;

            if ($request->link_type === 'custom_page') {
                $customPage = $request->input('custom_page');
            } elseif ($request->link_type === 'link_url') {
                $linkUrl = $request->input('link_url');
            }

            $menu = Menu::create([
                'title' => $request->title,
                'title_text' => $request->title_text,
                'link_type' => $request->link_type,
                'custom_page' => $customPage,
                'link_url' => $linkUrl,
                'order' => $request->order,
                'submenus' => json_encode([]),
                'is_active' => $request->input('is_active', 1),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Menu created successfully!',
                'menu' => $menu
            ]);
        } catch (\Exception $e) {
            Log::error('Quick Store Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create menu'
            ], 500);
        }
    }

    /**
     * Show edit form for menu
     */
    public function edit(Request $request)
    {
        try {
            $menu = Menu::findOrFail($request->id);
            $menuArray = $menu->toArray();
            $menuArray['submenus'] = !empty($menu->submenus) 
                ? json_decode($menu->submenus, true) 
                : [];

            $customPages = CustomPageModel::where('status', 1)->get();

            return view('content.admin.menu_management.edit', compact('menuArray', 'customPages'));
        } catch (\Exception $e) {
            Log::error('Menu Edit Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to load menu for editing.');
        }
    }

    /**
     * Update menu
     */
    public function update(Request $request)
    {
        try {
            $menu = Menu::findOrFail($request->id);

            $validator = Validator::make($request->all(), [
                'title' => 'required|string|max:255|unique:content_menus,title,' . $menu->id,
                'title_text' => 'required|string|max:255',
                'main_link_type' => 'required|in:custom_page,link_url,placeholder',
                'main_order' => 'required|integer|min:1',
            ]);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Validation failed.');
            }

            $customPage = null;
            $linkUrl = null;

            if ($request->main_link_type === 'custom_page') {
                $customPage = $request->input('main_custom_page');
            } elseif ($request->main_link_type === 'link_url') {
                $linkUrl = $request->input('main_link_url');
            }

            // Process submenus recursively
            $submenus = [];
            if ($request->has('submenus') && is_array($request->submenus)) {
                $submenus = $this->processSubmenus($request->submenus);
            }

            $menu->update([
                'title' => $request->title,
                'title_text' => $request->title_text,
                'link_type' => $request->main_link_type,
                'custom_page' => $customPage,
                'link_url' => $linkUrl,
                'order' => $request->main_order,
                'submenus' => json_encode($submenus),
                'is_active' => $request->has('is_active') ? 1 : 0,
            ]);

            return redirect()->route('admin.content-menu_management')->with('success', 'Menu updated successfully!');
        } catch (\Exception $e) {
            Log::error('Menu Update Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to update menu.')->withInput();
        }
    }

    /**
     * Quick update menu (AJAX)
     */
    public function quickUpdate(Request $request)
    {
        try {
            $menu = Menu::findOrFail($request->id);

            $updateData = [];

            if ($request->has('title')) {
                $updateData['title'] = $request->title;
            }
            if ($request->has('title_text')) {
                $updateData['title_text'] = $request->title_text;
            }
            if ($request->has('link_type')) {
                $updateData['link_type'] = $request->link_type;

                // Handle link based on type
                if ($request->link_type === 'custom_page') {
                    $updateData['custom_page'] = $request->input('custom_page');
                    $updateData['link_url'] = null;
                } elseif ($request->link_type === 'link_url') {
                    $updateData['link_url'] = $request->input('link_url');
                    $updateData['custom_page'] = null;
                } else {
                    $updateData['custom_page'] = null;
                    $updateData['link_url'] = null;
                }
            }
            if ($request->has('order')) {
                $updateData['order'] = $request->order;
            }
            if ($request->has('is_active')) {
                $updateData['is_active'] = $request->is_active;
            }

            $menu->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'Menu updated successfully!',
                'menu' => $menu
            ]);
        } catch (\Exception $e) {
            Log::error('Quick Update Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update menu'
            ], 500);
        }
    }

    /**
     * Delete menu
     */
    public function delete(Request $request)
    {
        try {
            $menu = Menu::findOrFail($request->id);

            // Count all submenus recursively
            $submenus = !empty($menu->submenus) ? json_decode($menu->submenus, true) : [];
            $submenuCount = $this->countAllSubmenus($submenus);

            $menu->delete();

            return response()->json([
                'success' => true,
                'message' => 'Menu deleted successfully!',
                'submenu_count' => $submenuCount
            ]);
        } catch (\Exception $e) {
            Log::error('Menu Delete Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete menu'
            ], 500);
        }
    }

    /**
     * Update menu status (Toggle active/inactive)
     */
    public function updateStatus(Request $request)
    {
        try {
            $menu = Menu::findOrFail($request->id);
            $menu->is_active = $request->input('is_active', !$menu->is_active);
            $menu->save();

            return response()->json([
                'success' => true,
                'message' => 'Menu status updated successfully!',
                'is_active' => $menu->is_active
            ]);
        } catch (\Exception $e) {
            Log::error('Update Status Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update status'
            ], 500);
        }
    }

    /**
     * Quick add submenu at any level (supports unlimited nesting)
     */
    public function quickAddSubmenu(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'parent_id' => 'required|exists:menus,id',
                'title' => 'required|string|max:255',
                'title_text' => 'required|string|max:255',
                'link_type' => 'required|in:custom_page,link_url,placeholder',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $menu = Menu::findOrFail($request->parent_id);
            $submenus = !empty($menu->submenus) ? json_decode($menu->submenus, true) : [];

            // Prepare new submenu
            $newSubmenu = [
                'title' => $request->title,
                'title_text' => $request->title_text,
                'link_type' => $request->link_type,
                'custom_page' => $request->link_type === 'custom_page' ? $request->input('custom_page') : null,
                'link_url' => $request->link_type === 'link_url' ? $request->input('link_url') : null,
                'has_submenu' => false,
                'submenus' => []
            ];

            // Navigate to correct parent using path (e.g., "0.2.1")
            $parentPath = $request->input('parent_path', '');
            
            if (empty($parentPath)) {
                // Add to root level
                $submenus[] = $newSubmenu;
            } else {
                // Navigate to nested parent
                $pathParts = explode('.', $parentPath);
                $current = &$submenus;
                
                foreach ($pathParts as $index) {
                    if (isset($current[$index])) {
                        $current = &$current[$index]['submenus'];
                    }
                }
                
                $current[] = $newSubmenu;
                
                // Update parent has_submenu flag
                $current = &$submenus;
                foreach ($pathParts as $index) {
                    if (isset($current[$index])) {
                        $current[$index]['has_submenu'] = true;
                        $current = &$current[$index]['submenus'];
                    }
                }
            }

            $menu->submenus = json_encode($submenus);
            $menu->save();

            return response()->json([
                'success' => true,
                'message' => 'Submenu added successfully!',
                'submenus' => $submenus
            ]);
        } catch (\Exception $e) {
            Log::error('Quick Add Submenu Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to add submenu'
            ], 500);
        }
    }

    /**
     * Update specific submenu at any nesting level
     */
    public function updateSubmenu(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'parent_id' => 'required|exists:menus,id',
                'submenu_path' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $menu = Menu::findOrFail($request->parent_id);
            $submenus = !empty($menu->submenus) ? json_decode($menu->submenus, true) : [];

            // Navigate to submenu using path
            $pathParts = explode('.', $request->submenu_path);
            $current = &$submenus;
            
            foreach ($pathParts as $i => $index) {
                if ($i === count($pathParts) - 1) {
                    // Update the target submenu
                    if (isset($current[$index])) {
                        if ($request->has('title')) {
                            $current[$index]['title'] = $request->title;
                        }
                        if ($request->has('title_text')) {
                            $current[$index]['title_text'] = $request->title_text;
                        }
                        if ($request->has('link_type')) {
                            $current[$index]['link_type'] = $request->link_type;
                            
                            if ($request->link_type === 'custom_page') {
                                $current[$index]['custom_page'] = $request->input('custom_page');
                                $current[$index]['link_url'] = null;
                            } elseif ($request->link_type === 'link_url') {
                                $current[$index]['link_url'] = $request->input('link_url');
                                $current[$index]['custom_page'] = null;
                            } else {
                                $current[$index]['custom_page'] = null;
                                $current[$index]['link_url'] = null;
                            }
                        }
                    }
                } else {
                    if (isset($current[$index]['submenus'])) {
                        $current = &$current[$index]['submenus'];
                    }
                }
            }

            $menu->submenus = json_encode($submenus);
            $menu->save();

            return response()->json([
                'success' => true,
                'message' => 'Submenu updated successfully!',
                'submenus' => $submenus
            ]);
        } catch (\Exception $e) {
            Log::error('Update Submenu Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update submenu'
            ], 500);
        }
    }

    /**
     * Delete submenu by path
     */
    public function deleteSubmenu(Request $request)
    {
        try {
            $menu = Menu::findOrFail($request->parent_id);
            $submenus = !empty($menu->submenus) ? json_decode($menu->submenus, true) : [];

            // Navigate to parent level
            $pathParts = explode('.', $request->submenu_path);
            $current = &$submenus;
            
            for ($i = 0; $i < count($pathParts) - 1; $i++) {
                if (isset($current[$pathParts[$i]]['submenus'])) {
                    $current = &$current[$pathParts[$i]]['submenus'];
                }
            }

            // Remove item at final index
            $finalIndex = end($pathParts);
            if (isset($current[$finalIndex])) {
                unset($current[$finalIndex]);
                
                // Re-index array
                $current = array_values($current);
            }

            // Update has_submenu flag if needed
            if (count($pathParts) > 1) {
                $parentCurrent = &$submenus;
                for ($i = 0; $i < count($pathParts) - 1; $i++) {
                    if ($i === count($pathParts) - 2) {
                        $parentCurrent[$pathParts[$i]]['has_submenu'] = !empty($parentCurrent[$pathParts[$i]]['submenus']);
                    } else {
                        $parentCurrent = &$parentCurrent[$pathParts[$i]]['submenus'];
                    }
                }
            }

            $menu->submenus = json_encode($submenus);
            $menu->save();

            return response()->json([
                'success' => true,
                'message' => 'Submenu deleted successfully!',
                'submenus' => $submenus
            ]);
        } catch (\Exception $e) {
            Log::error('Delete Submenu Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete submenu'
            ], 500);
        }
    }

    /**
     * Reorder main menus
     */
    public function reorderMenus(Request $request)
    {
        try {
            $menuIds = $request->input('menu_ids', []);

            foreach ($menuIds as $index => $menuId) {
                Menu::where('id', $menuId)->update(['order' => $index + 1]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Menus reordered successfully!'
            ]);
        } catch (\Exception $e) {
            Log::error('Reorder Menus Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to reorder menus'
            ], 500);
        }
    }

    /**
     * Reorder submenus with nesting support
     */
    public function reorderSubmenus(Request $request)
    {
        try {
            $menu = Menu::findOrFail($request->menu_id);
            $submenus = !empty($menu->submenus) ? json_decode($menu->submenus, true) : [];

            $submenuPaths = $request->input('submenu_paths', []);
            $parentPath = $request->input('parent_path', '');

            // Navigate to correct level
            $current = &$submenus;
            if (!empty($parentPath)) {
                $pathParts = explode('.', $parentPath);
                foreach ($pathParts as $index) {
                    if (isset($current[$index]['submenus'])) {
                        $current = &$current[$index]['submenus'];
                    }
                }
            }

            // Reorder based on submenu_paths array
            $reordered = [];
            foreach ($submenuPaths as $path) {
                $pathParts = explode('.', $path);
                $finalIndex = end($pathParts);
                if (isset($current[$finalIndex])) {
                    $reordered[] = $current[$finalIndex];
                }
            }

            $current = $reordered;

            $menu->submenus = json_encode($submenus);
            $menu->save();

            return response()->json([
                'success' => true,
                'message' => 'Submenus reordered successfully!',
                'submenus' => $submenus
            ]);
        } catch (\Exception $e) {
            Log::error('Reorder Submenus Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to reorder submenus'
            ], 500);
        }
    }

    /**
     * Move menu item between parents (supports menu to submenu, submenu to menu, submenu to different parent)
     */
    public function moveMenuItem(Request $request)
    {
        try {
            $itemType = $request->input('item_type'); // 'menu' or 'submenu'
            $itemId = $request->input('item_id');
            $itemPath = $request->input('item_path', '');
            $newParentId = $request->input('new_parent_id');
            $newParentPath = $request->input('new_parent_path', '');

            if ($itemType === 'menu') {
                // Moving main menu to become a submenu
                $sourceMenu = Menu::findOrFail($itemId);
                $targetMenu = Menu::findOrFail($newParentId);
                
                $targetSubmenus = !empty($targetMenu->submenus) ? json_decode($targetMenu->submenus, true) : [];
                
                // Prepare menu item as submenu
                $newSubmenu = [
                    'title' => $sourceMenu->title,
                    'title_text' => $sourceMenu->title_text,
                    'link_type' => $sourceMenu->link_type,
                    'custom_page' => $sourceMenu->custom_page,
                    'link_url' => $sourceMenu->link_url,
                    'has_submenu' => false,
                    'submenus' => !empty($sourceMenu->submenus) ? json_decode($sourceMenu->submenus, true) : []
                ];
                
                // Update has_submenu if it has submenus
                if (!empty($newSubmenu['submenus'])) {
                    $newSubmenu['has_submenu'] = true;
                    $newSubmenu['submenus'] = $this->preserveNestedSubmenus($newSubmenu['submenus']);
                }
                
                // Add to target
                if (empty($newParentPath)) {
                    $targetSubmenus[] = $newSubmenu;
                } else {
                    $pathParts = explode('.', $newParentPath);
                    $current = &$targetSubmenus;
                    foreach ($pathParts as $index) {
                        if (isset($current[$index]['submenus'])) {
                            $current = &$current[$index]['submenus'];
                        }
                    }
                    $current[] = $newSubmenu;
                }
                
                $targetMenu->submenus = json_encode($targetSubmenus);
                $targetMenu->save();
                
                // Delete source menu
                $sourceMenu->delete();
                
            } else {
                // Moving submenu
                $sourceMenu = Menu::findOrFail($itemId);
                $sourceSubmenus = !empty($sourceMenu->submenus) ? json_decode($sourceMenu->submenus, true) : [];
                
                // Extract item from source
                $pathParts = explode('.', $itemPath);
                $current = &$sourceSubmenus;
                
                for ($i = 0; $i < count($pathParts) - 1; $i++) {
                    if (isset($current[$pathParts[$i]]['submenus'])) {
                        $current = &$current[$pathParts[$i]]['submenus'];
                    }
                }
                
                $finalIndex = end($pathParts);
                $movingItem = $current[$finalIndex];
                unset($current[$finalIndex]);
                $current = array_values($current);
                
                // Save source
                $sourceMenu->submenus = json_encode($sourceSubmenus);
                $sourceMenu->save();
                
                // Add to target
                if ($newParentId === 'root') {
                    // Moving submenu to become main menu
                    Menu::create([
                        'title' => $movingItem['title'],
                        'title_text' => $movingItem['title_text'],
                        'link_type' => $movingItem['link_type'],
                        'custom_page' => $movingItem['custom_page'] ?? null,
                        'link_url' => $movingItem['link_url'] ?? null,
                        'order' => Menu::max('order') + 1,
                        'submenus' => json_encode($movingItem['submenus'] ?? []),
                        'is_active' => 1,
                    ]);
                } else {
                    // Moving to another submenu
                    $targetMenu = Menu::findOrFail($newParentId);
                    $targetSubmenus = !empty($targetMenu->submenus) ? json_decode($targetMenu->submenus, true) : [];
                    
                    if (empty($newParentPath)) {
                        $targetSubmenus[] = $movingItem;
                    } else {
                        $pathParts = explode('.', $newParentPath);
                        $current = &$targetSubmenus;
                        foreach ($pathParts as $index) {
                            if (isset($current[$index]['submenus'])) {
                                $current = &$current[$index]['submenus'];
                            }
                        }
                        $current[] = $movingItem;
                    }
                    
                    $targetMenu->submenus = json_encode($targetSubmenus);
                    $targetMenu->save();
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Menu item moved successfully!'
            ]);
        } catch (\Exception $e) {
            Log::error('Move Menu Item Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to move menu item'
            ], 500);
        }
    }

    /**
     * Search menus
     */
    public function search(Request $request)
    {
        try {
            $query = Menu::query();

            if ($request->has('search') && !empty($request->search)) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                      ->orWhere('title_text', 'like', "%{$search}%")
                      ->orWhere('custom_page', 'like', "%{$search}%")
                      ->orWhere('link_url', 'like', "%{$search}%");
                });
            }

            if ($request->has('status') && $request->status !== '') {
                $query->where('is_active', $request->status);
            }

            $menus = $query->orderBy('order')->get();

            return response()->json([
                'success' => true,
                'menus' => $menus
            ]);
        } catch (\Exception $e) {
            Log::error('Search Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Search failed'
            ], 500);
        }
    }

    /**
     * Check if order number is unique
     */
    public function checkOrderNumber(Request $request)
    {
        try {
            $order = $request->input('order');
            $excludeId = $request->input('exclude_id');

            $query = Menu::where('order', $order);

            if ($excludeId) {
                $query->where('id', '!=', $excludeId);
            }

            $exists = $query->exists();

            return response()->json([
                'valid' => !$exists,
                'message' => $exists ? 'Order number already exists' : 'Order number is available'
            ]);
        } catch (\Exception $e) {
            Log::error('Check Order Number Error: ' . $e->getMessage());
            return response()->json([
                'valid' => false,
                'message' => 'Failed to check order number'
            ], 500);
        }
    }

    /**
     * Get single menu for editing (AJAX)
     */
    public function getMenu($id)
    {
        try {
            $menu = Menu::findOrFail($id);
            $submenus = !empty($menu->submenus) ? json_decode($menu->submenus, true) : [];
            
            // Clean submenus
            $submenus = $this->cleanSubmenus($submenus);

            return response()->json([
                'success' => true,
                'menu' => [
                    'id' => $menu->id,
                    'title' => $menu->title,
                    'title_text' => $menu->title_text,
                    'link_type' => $menu->link_type,
                    'custom_page' => $menu->custom_page,
                    'link_url' => $menu->link_url,
                    'order' => $menu->order,
                    'is_active' => $menu->is_active,
                    'submenus' => $submenus,
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Get Menu Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to get menu'
            ], 500);
        }
    }

    /**
     * Process submenus recursively
     * 
     * @param array $submenus
     * @return array
     */
    private function processSubmenus($submenus)
    {
        $processed = [];

        foreach ($submenus as $submenu) {
            if (!is_array($submenu)) {
                continue;
            }

            // Validate required fields
            if (empty($submenu['title']) || empty($submenu['title_text']) || empty($submenu['link_type'])) {
                continue;
            }

            $processedSubmenu = [
                'title' => $submenu['title'],
                'title_text' => $submenu['title_text'],
                'link_type' => $submenu['link_type'],
                'custom_page' => null,
                'link_url' => null,
                'has_submenu' => false,
                'submenus' => []
            ];

            // Handle link type
            if ($submenu['link_type'] === 'custom_page') {
                $processedSubmenu['custom_page'] = $submenu['custom_page'] ?? null;
            } elseif ($submenu['link_type'] === 'link_url') {
                $processedSubmenu['link_url'] = $submenu['link_url'] ?? null;
            }

            // Process nested submenus recursively
            if (isset($submenu['submenus']) && is_array($submenu['submenus']) && !empty($submenu['submenus'])) {
                $processedSubmenu['submenus'] = $this->processSubmenus($submenu['submenus']);
                $processedSubmenu['has_submenu'] = !empty($processedSubmenu['submenus']);
            }

            $processed[] = $processedSubmenu;
        }

        return $processed;
    }

    /**
     * Count all submenus recursively
     * 
     * @param array $submenus
     * @return int
     */
    private function countAllSubmenus($submenus)
    {
        $count = count($submenus);

        foreach ($submenus as $submenu) {
            if (isset($submenu['submenus']) && is_array($submenu['submenus']) && !empty($submenu['submenus'])) {
                $count += $this->countAllSubmenus($submenu['submenus']);
            }
        }

        return $count;
    }

    /**
     * Clean invalid submenu entries
     * 
     * @param array $submenus
     * @return array
     */
    private function cleanSubmenus($submenus)
    {
        if (!is_array($submenus)) {
            return [];
        }

        $cleaned = [];

        foreach ($submenus as $submenu) {
            if (!is_array($submenu)) {
                continue;
            }

            // Fix missing link_type (backward compatibility)
            if (!isset($submenu['link_type'])) {
                if (!empty($submenu['custom_page'])) {
                    $submenu['link_type'] = 'custom_page';
                } elseif (!empty($submenu['link_url'])) {
                    $submenu['link_type'] = 'link_url';
                } else {
                    $submenu['link_type'] = 'placeholder';
                }
            }

            // Ensure submenus is always array
            if (!isset($submenu['submenus']) || !is_array($submenu['submenus'])) {
                $submenu['submenus'] = [];
            }

            // Recursively clean nested submenus
            if (!empty($submenu['submenus'])) {
                $submenu['submenus'] = $this->cleanSubmenus($submenu['submenus']);
                $submenu['has_submenu'] = !empty($submenu['submenus']);
            } else {
                $submenu['has_submenu'] = false;
            }

            $cleaned[] = $submenu;
        }

        return $cleaned;
    }

    /**
     * Preserve nested submenu structure when moving
     * 
     * @param array $submenus
     * @return array
     */
    private function preserveNestedSubmenus($submenus)
    {
        $preserved = [];

        foreach ($submenus as $submenu) {
            if (!is_array($submenu)) {
                continue;
            }

            $preservedSubmenu = [
                'title' => $submenu['title'] ?? '',
                'title_text' => $submenu['title_text'] ?? '',
                'link_type' => $submenu['link_type'] ?? 'placeholder',
                'custom_page' => $submenu['custom_page'] ?? null,
                'link_url' => $submenu['link_url'] ?? null,
                'has_submenu' => false,
                'submenus' => []
            ];

            // Recursively preserve nested submenus
            if (isset($submenu['submenus']) && is_array($submenu['submenus']) && !empty($submenu['submenus'])) {
                $preservedSubmenu['submenus'] = $this->preserveNestedSubmenus($submenu['submenus']);
                $preservedSubmenu['has_submenu'] = !empty($preservedSubmenu['submenus']);
            }

            $preserved[] = $preservedSubmenu;
        }

        return $preserved;
    }
}
