<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AgreementHistory;
use App\Models\Deposit;
use App\Models\DepositReturn;
use App\Models\Rent;
use App\Models\Student;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Mail;
use Illuminate\View\View;
use Symfony\Component\HttpFoundation\Response;

class PdfController extends Controller
{
    /**
     * Download agreement as PDF.
     */
    public function agreement(Student $student): Response
    {
        $student->load(['bed', 'room', 'floor', 'documents', 'agreementHistories']);
        $firstAgreement = $student->agreementHistories->sortBy('from_date')->first();
        $latestAgreement = $student->agreementHistories->sortByDesc('to_date')->first();
        $admissionDate = $firstAgreement?->from_date;
        $leavingDate = $latestAgreement?->to_date;
        $baseUrl = url('/');

        $pdf = Pdf::loadView('pdf.agreement', compact('student', 'admissionDate', 'leavingDate', 'baseUrl'))
            ->setPaper('a4', 'portrait')
            ->setOption('isRemoteEnabled', true);

        $filename = 'agreement-' . preg_replace('/[^a-z0-9-]/', '-', strtolower($student->full_name)) . '-' . now()->format('Y-m-d') . '.pdf';

        return $pdf->download($filename);
    }

    /**
     * Rent receipt as PDF. Uses cached file if it exists; otherwise generates and saves (rent receipt only).
     */
    public function rentReceipt(Rent $rent): Response
    {
        $dir = storage_path('app/receipts/rent');
        if (! File::isDirectory($dir)) {
            File::makeDirectory($dir, 0755, true);
        }
        $path = $dir . '/' . $rent->id . '.pdf';

        if (! File::exists($path)) {
            $rent->load(['student.bed', 'student.room', 'student.floor']);
            $amountInWords = $this->amountToWords((float) $rent->amount);
            $pdf = Pdf::loadView('pdf.rent-receipt', compact('rent', 'amountInWords'))
                ->setPaper('a4', 'portrait');
            $pdf->save($path);
        }

        $filename = 'rent-receipt-' . ($rent->receipt_no ?? 'R' . $rent->id) . '.pdf';

        return response()->file($path, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'inline; filename="' . $filename . '"',
        ]);
    }

    /**
     * Download deposit receipt as PDF.
     */
    public function depositReceipt(Deposit $deposit): Response
    {
        $deposit->load(['student.bed', 'student.room', 'student.floor']);
        $amountInWords = $this->amountToWords((float) $deposit->amount);

        $pdf = Pdf::loadView('pdf.deposit-receipt', compact('deposit', 'amountInWords'))
            ->setPaper('a4', 'portrait');

        return $pdf->download('deposit-receipt-D-' . $deposit->id . '.pdf');
    }

    /**
     * Download deposit return receipt as PDF.
     */
    public function depositReturnReceipt(DepositReturn $depositReturn): Response
    {
        $depositReturn->load(['student.bed', 'student.room', 'student.floor']);
        $amountInWords = $this->amountToWords((float) $depositReturn->amount_returned);

        $pdf = Pdf::loadView('pdf.deposit-return-receipt', compact('depositReturn', 'amountInWords'))
            ->setPaper('a4', 'portrait');

        return $pdf->download('deposit-return-receipt-DR-' . $depositReturn->id . '.pdf');
    }

    /**
     * Send document as PDF attachment to an email address (POST).
     */
    public function sendEmail(Request $request)
    {
        $request->validate([
            'type' => 'required|in:agreement,rent,deposit,deposit_return',
            'email' => 'required|email',
            'id' => 'required|integer',
        ]);

        $type = $request->input('type');
        $email = $request->input('email');
        $id = (int) $request->input('id');

        $pdf = null;
        $filename = '';
        $subject = '';

        switch ($type) {
            case 'agreement':
                $student = Student::findOrFail($id);
                $student->load(['bed', 'room', 'floor', 'documents', 'agreementHistories']);
                $firstAgreement = $student->agreementHistories->sortBy('from_date')->first();
                $latestAgreement = $student->agreementHistories->sortByDesc('to_date')->first();
                $admissionDate = $firstAgreement?->from_date;
                $leavingDate = $latestAgreement?->to_date;
                $baseUrl = url('/');
                $pdf = Pdf::loadView('pdf.agreement', compact('student', 'admissionDate', 'leavingDate', 'baseUrl'))->setPaper('a4', 'portrait');
                $filename = 'agreement-' . $student->full_name . '.pdf';
                $subject = 'Agreement - ' . $student->full_name;
                break;
            case 'rent':
                $rent = Rent::with(['student.bed', 'student.room', 'student.floor'])->findOrFail($id);
                $amountInWords = $this->amountToWords((float) $rent->amount);
                $pdf = Pdf::loadView('pdf.rent-receipt', compact('rent', 'amountInWords'))->setPaper('a4', 'portrait');
                $filename = 'rent-receipt-' . ($rent->receipt_no ?? 'R' . $rent->id) . '.pdf';
                $subject = 'Rent Receipt - ' . ($rent->receipt_no ?? 'R' . $rent->id);
                break;
            case 'deposit':
                $deposit = Deposit::with(['student.bed', 'student.room', 'student.floor'])->findOrFail($id);
                $amountInWords = $this->amountToWords((float) $deposit->amount);
                $pdf = Pdf::loadView('pdf.deposit-receipt', compact('deposit', 'amountInWords'))->setPaper('a4', 'portrait');
                $filename = 'deposit-receipt-D-' . $deposit->id . '.pdf';
                $subject = 'Deposit Receipt - D-' . $deposit->id;
                break;
            case 'deposit_return':
                $depositReturn = DepositReturn::with(['student.bed', 'student.room', 'student.floor'])->findOrFail($id);
                $amountInWords = $this->amountToWords((float) $depositReturn->amount_returned);
                $pdf = Pdf::loadView('pdf.deposit-return-receipt', compact('depositReturn', 'amountInWords'))->setPaper('a4', 'portrait');
                $filename = 'deposit-return-receipt-DR-' . $depositReturn->id . '.pdf';
                $subject = 'Deposit Return Receipt - DR-' . $depositReturn->id;
                break;
        }

        $pdfContent = $pdf->output();

        try {
            Mail::raw('Please find the attached document.', function ($message) use ($email, $subject, $filename, $pdfContent) {
                $message->to($email)
                    ->subject($subject)
                    ->attachData($pdfContent, $filename, ['mime' => 'application/pdf']);
            });
        } catch (\Throwable $e) {
            return response()->json(['success' => false, 'message' => 'Could not send email. Check mail configuration.'], 500);
        }

        return response()->json(['success' => true, 'message' => 'Document sent to ' . $email]);
    }

    private function amountToWords(float $amount): string
    {
        $rupees = (int) floor($amount);
        $paise = (int) round(($amount - $rupees) * 100);
        $words = $this->numberToWords($rupees);
        $out = $words ? ucfirst($words) . ' Rupees' : 'Zero Rupees';
        if ($paise > 0) {
            $out .= ' and ' . ucfirst($this->numberToWords($paise)) . ' Paise';
        }
        return $out . ' only';
    }

    private function numberToWords(int $n): string
    {
        if ($n === 0) {
            return '';
        }
        $ones = ['', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine', 'ten', 'eleven', 'twelve', 'thirteen', 'fourteen', 'fifteen', 'sixteen', 'seventeen', 'eighteen', 'nineteen'];
        $tens = ['', '', 'twenty', 'thirty', 'forty', 'fifty', 'sixty', 'seventy', 'eighty', 'ninety'];
        if ($n < 20) {
            return $ones[$n];
        }
        if ($n < 100) {
            return $tens[(int) floor($n / 10)] . ($n % 10 ? ' ' . $ones[$n % 10] : '');
        }
        if ($n < 1000) {
            return $ones[(int) floor($n / 100)] . ' hundred' . ($n % 100 ? ' ' . $this->numberToWords($n % 100) : '');
        }
        if ($n < 100000) {
            return $this->numberToWords((int) floor($n / 1000)) . ' thousand' . ($n % 1000 ? ' ' . $this->numberToWords($n % 1000) : '');
        }
        if ($n < 10000000) {
            return $this->numberToWords((int) floor($n / 100000)) . ' lakh' . ($n % 100000 ? ' ' . $this->numberToWords($n % 100000) : '');
        }
        return $this->numberToWords((int) floor($n / 10000000)) . ' crore' . ($n % 10000000 ? ' ' . $this->numberToWords($n % 10000000) : '');
    }
}
