<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WebsiteFeature;
use App\Models\WebsiteSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class WebsiteController extends Controller
{
    /**
     * Show hero section settings.
     */
    public function hero()
    {
        $settings = WebsiteSetting::getSetting();
        return view('admin.website.hero', compact('settings'));
    }

    /**
     * Update hero section.
     */
    public function updateHero(Request $request)
    {
        $settings = WebsiteSetting::getSetting();

        $validated = $request->validate([
            'hero_title' => ['required', 'string', 'max:255'],
            'hero_subtitle' => ['nullable', 'string'],
            'hero_background_color' => ['nullable', 'string', 'max:20'],
            'hero_button_text' => ['required', 'string', 'max:100'],
            'hero_button_link' => ['required', 'string', 'max:255'],
            'hero_background_image' => ['nullable', 'image', 'mimes:jpg,jpeg,png', 'max:2048'],
        ]);

        $data = [
            'hero_title' => $validated['hero_title'],
            'hero_subtitle' => $validated['hero_subtitle'] ?? null,
            'hero_background_color' => $validated['hero_background_color'] ?? '#667eea',
            'hero_button_text' => $validated['hero_button_text'],
            'hero_button_link' => $validated['hero_button_link'],
        ];

        // Handle background image upload
        if ($request->hasFile('hero_background_image')) {
            if ($settings->hero_background_image) {
                Storage::disk('public')->delete($settings->hero_background_image);
            }
            $data['hero_background_image'] = $request->file('hero_background_image')->store('website', 'public');
        }

        // Handle image removal
        if ($request->input('remove_background') === '1' && $settings->hero_background_image) {
            Storage::disk('public')->delete($settings->hero_background_image);
            $data['hero_background_image'] = null;
        }

        $settings->update($data);

        return back()->with('success', 'Hero section updated successfully!');
    }

    /**
     * Show features list.
     */
    public function features()
    {
        $features = WebsiteFeature::ordered()->get();
        return view('admin.website.features', compact('features'));
    }

    /**
     * Store new feature.
     */
    public function storeFeature(Request $request)
    {
        $validated = $request->validate([
            'icon' => ['required', 'string', 'max:100'],
            'title' => ['required', 'string', 'max:255'],
            'description' => ['required', 'string'],
            'sort_order' => ['nullable', 'integer'],
        ]);

        $validated['sort_order'] = $validated['sort_order'] ?? WebsiteFeature::max('sort_order') + 1;

        WebsiteFeature::create($validated);

        return back()->with('success', 'Feature added successfully!');
    }

    /**
     * Update feature.
     */
    public function updateFeature(Request $request, WebsiteFeature $feature)
    {
        $validated = $request->validate([
            'icon' => ['required', 'string', 'max:100'],
            'title' => ['required', 'string', 'max:255'],
            'description' => ['required', 'string'],
            'sort_order' => ['required', 'integer'],
            'is_active' => ['sometimes', 'boolean'],
        ]);

        $validated['is_active'] = $request->has('is_active');

        $feature->update($validated);

        return back()->with('success', 'Feature updated successfully!');
    }

    /**
     * Delete feature.
     */
    public function deleteFeature(WebsiteFeature $feature)
    {
        $feature->delete();
        return back()->with('success', 'Feature deleted successfully!');
    }

    /**
     * Show about section settings.
     */
    public function about()
    {
        $settings = WebsiteSetting::getSetting();
        return view('admin.website.about', compact('settings'));
    }

    /**
     * Update about section.
     */
    public function updateAbout(Request $request)
    {
        $settings = WebsiteSetting::getSetting();

        $validated = $request->validate([
            'about_title' => ['required', 'string', 'max:255'],
            'about_description' => ['required', 'string'],
            'stat_residents' => ['required', 'integer', 'min:0'],
            'stat_experience' => ['required', 'integer', 'min:0'],
        ]);

        $settings->update($validated);

        return back()->with('success', 'About section updated successfully!');
    }

    /**
     * Show general settings.
     */
    public function general()
    {
        $settings = WebsiteSetting::getSetting();
        return view('admin.website.general', compact('settings'));
    }

    /**
     * Update general settings.
     */
    public function updateGeneral(Request $request)
    {
        $settings = WebsiteSetting::getSetting();

        $validated = $request->validate([
            'primary_color' => ['required', 'string', 'max:20'],
            'secondary_color' => ['required', 'string', 'max:20'],
            'meta_description' => ['nullable', 'string'],
            'meta_keywords' => ['nullable', 'string'],
            'custom_css' => ['nullable', 'string'],
            'custom_js' => ['nullable', 'string'],
        ]);

        $settings->update($validated);

        return back()->with('success', 'General settings updated successfully!');
    }
}
