<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use App\Models\CustomPageModel;
use App\Helpers\MenuHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class MenuManagementController extends Controller
{
    /**
     * Display a listing of menus
     */
    public function index()
    {
        $menus = Menu::active()->ordered()->get();
        
        // Convert to array format for filtering
        $menusArray = $menus->map(function ($menu) {
            return [
                'id' => $menu->id,
                'title' => $menu->title,
                'title_text' => $menu->title_text,
                'link_type' => $menu->link_type,
                'custom_page' => $menu->custom_page,
                'link_url' => $menu->link_url,
                'link' => $menu->link,
                'order' => $menu->order,
                'is_active' => $menu->is_active,
                'submenus' => $menu->submenus_array,
                'submenu_count' => $menu->countSubmenus(),
                'depth' => $menu->getDepth(),
            ];
        })->toArray();
        
        // Filter by user permissions
        $filteredMenus = MenuHelper::filterMenuByPermissions($menusArray);
        
        return view('menu.index', compact('filteredMenus', 'menus'));
    }

    /**
     * Display all menus (admin view without filtering)
     */
    public function adminIndex()
    {
        $menus = Menu::ordered()->get();
        
        return view('menu.admin-index', compact('menus'));
    }

    /**
     * Show the form for creating a new menu
     */
    public function create()
    {
        $customPages = CustomPageModel::active()->get();
        
        return view('menu.create', compact('customPages'));
    }

    /**
     * Store a newly created menu
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255|unique:menus,title',
            'title_text' => 'required|string|max:255',
            'link_type' => 'required|in:custom_page,link_url,placeholder',
            'custom_page' => 'nullable|required_if:link_type,custom_page|string|max:255',
            'link_url' => 'nullable|required_if:link_type,link_url|url',
            'order' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $menu = Menu::create([
            'title' => $request->title,
            'title_text' => $request->title_text,
            'link_type' => $request->link_type,
            'custom_page' => $request->custom_page,
            'link_url' => $request->link_url,
            'order' => $request->order,
            'submenus' => json_encode([]),
            'is_active' => $request->is_active ?? true,
        ]);

        return redirect()->route('menu.index')
            ->with('success', 'Menu created successfully!');
    }

    /**
     * Display the specified menu
     */
    public function show(Menu $menu)
    {
        $menuData = [
            'id' => $menu->id,
            'title' => $menu->title,
            'title_text' => $menu->title_text,
            'link_type' => $menu->link_type,
            'custom_page' => $menu->custom_page,
            'link_url' => $menu->link_url,
            'link' => $menu->link,
            'order' => $menu->order,
            'is_active' => $menu->is_active,
            'submenus' => $menu->submenus_array,
            'submenu_count' => $menu->countSubmenus(),
            'depth' => $menu->getDepth(),
            'has_submenus' => $menu->hasSubmenus(),
            'created_at' => $menu->created_at,
            'updated_at' => $menu->updated_at,
        ];
        
        return view('menu.show', compact('menuData'));
    }

    /**
     * Show the form for editing the specified menu
     */
    public function edit(Menu $menu)
    {
        $customPages = CustomPageModel::active()->get();
        
        return view('menu.edit', compact('menu', 'customPages'));
    }

    /**
     * Update the specified menu
     */
    public function update(Request $request, Menu $menu)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255|unique:menus,title,' . $menu->id,
            'title_text' => 'required|string|max:255',
            'link_type' => 'required|in:custom_page,link_url,placeholder',
            'custom_page' => 'nullable|required_if:link_type,custom_page|string|max:255',
            'link_url' => 'nullable|required_if:link_type,link_url|url',
            'order' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $menu->update([
            'title' => $request->title,
            'title_text' => $request->title_text,
            'link_type' => $request->link_type,
            'custom_page' => $request->custom_page,
            'link_url' => $request->link_url,
            'order' => $request->order,
            'is_active' => $request->is_active ?? true,
        ]);

        return redirect()->route('menu.index')
            ->with('success', 'Menu updated successfully!');
    }

    /**
     * Update menu submenus
     */
    public function updateSubmenus(Request $request, Menu $menu)
    {
        $validator = Validator::make($request->all(), [
            'submenus' => 'required|json',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        $submenusArray = json_decode($request->submenus, true);
        $menu->submenus_array = $submenusArray;
        $menu->save();

        return response()->json([
            'success' => true,
            'message' => 'Submenus updated successfully!',
            'submenu_count' => $menu->countSubmenus(),
            'depth' => $menu->getDepth(),
        ]);
    }

    /**
     * Toggle menu active status
     */
    public function toggleActive(Menu $menu)
    {
        $menu->is_active = !$menu->is_active;
        $menu->save();

        return redirect()->back()
            ->with('success', 'Menu status updated successfully!');
    }

    /**
     * Remove the specified menu
     */
    public function destroy(Menu $menu)
    {
        $menu->delete();

        return redirect()->route('menu.index')
            ->with('success', 'Menu deleted successfully!');
    }

    /**
     * Get filtered menu for API/AJAX
     */
    public function getFilteredMenu()
    {
        $menus = Menu::active()->ordered()->get();
        
        $menusArray = $menus->map(function ($menu) {
            return [
                'id' => $menu->id,
                'title' => $menu->title,
                'title_text' => $menu->title_text,
                'link_type' => $menu->link_type,
                'custom_page' => $menu->custom_page,
                'link_url' => $menu->link_url,
                'link' => $menu->link,
                'submenus' => $menu->submenus_array,
            ];
        })->toArray();
        
        $filteredMenus = MenuHelper::filterMenuByPermissions($menusArray);
        
        return response()->json([
            'success' => true,
            'menus' => $filteredMenus,
        ]);
    }

    /**
     * Check if user can access a specific menu item
     */
    public function checkAccess(Request $request)
    {
        $menuItem = $request->all();
        $canAccess = MenuHelper::canAccessMenuItem($menuItem);
        
        return response()->json([
            'can_access' => $canAccess,
        ]);
    }
}
