<?php

namespace App\Http\Controllers;

use App\Models\AgreementHistory;
use App\Models\Deposit;
use App\Models\DepositReturn;
use App\Models\Rent;
use App\Models\Student;
use Illuminate\View\View;

class ShareController extends Controller
{
    /**
     * Public signed view: Agreement for a student (no login required).
     */
    public function agreement(Student $student): View
    {
        $student->load(['bed', 'room', 'floor', 'documents', 'agreementHistories']);
        $firstAgreement = $student->agreementHistories->sortBy('from_date')->first();
        $latestAgreement = $student->agreementHistories->sortByDesc('to_date')->first();
        $admissionDate = $firstAgreement?->from_date;
        $leavingDate = $latestAgreement?->to_date;

        return view('admin.agreements.view', [
            'student' => $student,
            'admissionDate' => $admissionDate,
            'leavingDate' => $leavingDate,
            'shareMode' => true,
        ]);
    }

    /**
     * Public signed view: Rent receipt (no login required).
     */
    public function rentReceipt(Rent $rent): View
    {
        $rent->load(['student.bed', 'student.room', 'student.floor']);
        $amountInWords = $this->amountToWords((float) $rent->amount);

        return view('admin.rents.receipt', [
            'rent' => $rent,
            'amountInWords' => $amountInWords,
            'shareMode' => true,
        ]);
    }

    /**
     * Public signed view: Deposit receipt (no login required).
     */
    public function depositReceipt(Deposit $deposit): View
    {
        $deposit->load(['student.bed', 'student.room', 'student.floor']);
        $amountInWords = $this->amountToWords((float) $deposit->amount);

        return view('admin.deposits.receipt', [
            'deposit' => $deposit,
            'amountInWords' => $amountInWords,
            'shareMode' => true,
        ]);
    }

    /**
     * Public signed view: Deposit return receipt (no login required).
     */
    public function depositReturnReceipt(DepositReturn $depositReturn): View
    {
        $depositReturn->load(['student.bed', 'student.room', 'student.floor']);
        $amountInWords = $this->amountToWords((float) $depositReturn->amount_returned);

        return view('admin.deposits.return-receipt', [
            'depositReturn' => $depositReturn,
            'amountInWords' => $amountInWords,
            'shareMode' => true,
        ]);
    }

    private function amountToWords(float $amount): string
    {
        $rupees = (int) floor($amount);
        $paise = (int) round(($amount - $rupees) * 100);
        $words = $this->numberToWords($rupees);
        $out = $words ? ucfirst($words) . ' Rupees' : 'Zero Rupees';
        if ($paise > 0) {
            $out .= ' and ' . ucfirst($this->numberToWords($paise)) . ' Paise';
        }
        return $out . ' only';
    }

    private function numberToWords(int $n): string
    {
        if ($n === 0) {
            return '';
        }
        $ones = ['', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine', 'ten', 'eleven', 'twelve', 'thirteen', 'fourteen', 'fifteen', 'sixteen', 'seventeen', 'eighteen', 'nineteen'];
        $tens = ['', '', 'twenty', 'thirty', 'forty', 'fifty', 'sixty', 'seventy', 'eighty', 'ninety'];
        if ($n < 20) {
            return $ones[$n];
        }
        if ($n < 100) {
            return $tens[(int) floor($n / 10)] . ($n % 10 ? ' ' . $ones[$n % 10] : '');
        }
        if ($n < 1000) {
            return $ones[(int) floor($n / 100)] . ' hundred' . ($n % 100 ? ' ' . $this->numberToWords($n % 100) : '');
        }
        if ($n < 100000) {
            return $this->numberToWords((int) floor($n / 1000)) . ' thousand' . ($n % 1000 ? ' ' . $this->numberToWords($n % 1000) : '');
        }
        if ($n < 10000000) {
            return $this->numberToWords((int) floor($n / 100000)) . ' lakh' . ($n % 100000 ? ' ' . $this->numberToWords($n % 100000) : '');
        }
        return $this->numberToWords((int) floor($n / 10000000)) . ' crore' . ($n % 10000000 ? ' ' . $this->numberToWords($n % 10000000) : '');
    }
}
