<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Menu extends Model
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'content_menus';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'title',
        'title_text',
        'link_type',
        'custom_page',
        'link_url',
        'order',
        'submenus',
        'is_active',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_active' => 'boolean',
        'order' => 'integer',
    ];

    /**
     * Get submenus as array
     *
     * @return array
     */
    public function getSubmenusArrayAttribute(): array
    {
        if (empty($this->submenus)) {
            return [];
        }

        $decoded = json_decode($this->submenus, true);
        return is_array($decoded) ? $decoded : [];
    }

    /**
     * Set submenus from array
     *
     * @param array $value
     * @return void
     */
    public function setSubmenusArrayAttribute(array $value): void
    {
        $this->attributes['submenus'] = json_encode($value);
    }

    /**
     * Count total submenus including all nested levels recursively
     *
     * @return int
     */
    public function countSubmenus(): int
    {
        $submenusArray = $this->submenus_array;
        
        if (empty($submenusArray)) {
            return 0;
        }

        return $this->countNestedItems($submenusArray);
    }

    /**
     * Get maximum nesting depth of submenus
     *
     * @return int
     */
    public function getDepth(): int
    {
        $submenusArray = $this->submenus_array;
        
        if (empty($submenusArray)) {
            return 0;
        }

        return $this->getMaxDepth($submenusArray);
    }

    /**
     * Check if menu has any submenus
     *
     * @return bool
     */
    public function hasSubmenus(): bool
    {
        $submenusArray = $this->submenus_array;
        return !empty($submenusArray);
    }

    /**
     * Private recursive method to count nested items
     *
     * @param array $items
     * @return int
     */
    private function countNestedItems(array $items): int
    {
        $count = count($items);

        foreach ($items as $item) {
            if (isset($item['submenus']) && is_array($item['submenus']) && !empty($item['submenus'])) {
                $count += $this->countNestedItems($item['submenus']);
            }
        }

        return $count;
    }

    /**
     * Private recursive method to get max depth
     *
     * @param array $items
     * @param int $currentDepth
     * @return int
     */
    private function getMaxDepth(array $items, int $currentDepth = 0): int
    {
        if (empty($items)) {
            return $currentDepth;
        }

        $maxDepth = $currentDepth + 1;

        foreach ($items as $item) {
            if (isset($item['submenus']) && is_array($item['submenus']) && !empty($item['submenus'])) {
                $depth = $this->getMaxDepth($item['submenus'], $currentDepth + 1);
                $maxDepth = max($maxDepth, $depth);
            }
        }

        return $maxDepth;
    }

    /**
     * Get the link for this menu item
     *
     * @return string|null
     */
    public function getLinkAttribute(): ?string
    {
        if (!empty($this->link_url)) {
            return $this->link_url;
        }

        if (!empty($this->custom_page)) {
            return $this->custom_page;
        }

        return null;
    }

    /**
     * Scope a query to only include active menus.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', 1);
    }

    /**
     * Scope a query to order menus by order column.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('order', 'asc');
    }
}
