<?php

namespace App\Services;

use App\Models\AgreementHistory;
use App\Models\Rent;
use App\Models\Student;
use App\Models\Deposit;
use App\Models\AdminNotification;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class NotificationService
{
    /**
     * Sync all notifications to database
     */
    public function syncNotifications(): void
    {
        // Clear old notifications (older than 30 days and read)
        AdminNotification::where('is_read', true)
            ->where('created_at', '<', Carbon::now()->subDays(30))
            ->delete();

        // Sync each type
        $this->syncExpiringAgreements();
        $this->syncOverdueRents();
        $this->syncDueRents();
        $this->syncLowDeposits();
        $this->syncPendingStudents();
    }

    /**
     * Get unread count
     */
    public function getUnreadCount(): int
    {
        return AdminNotification::unread()->count();
    }

    /**
     * Get all counts in one query (unread, critical, total) - fast for index/dropdown
     */
    public function getCounts(): array
    {
        $total = AdminNotification::count();
        $unread = AdminNotification::unread()->count();
        $critical = AdminNotification::where('priority', 'critical')->count();

        return [
            'unread' => $unread,
            'critical' => $critical,
            'total' => $total,
        ];
    }

    /**
     * Get category counts only (type + count) for notification bar - single fast query
     */
    public function getCategoryCounts(): Collection
    {
        $counts = AdminNotification::query()
            ->selectRaw('type, count(*) as count')
            ->groupBy('type')
            ->pluck('count', 'type');

        $labels = [
            'agreement_expiring' => 'Agreement expiring',
            'rent_due' => 'Rent due',
            'rent_overdue' => 'Rent overdue',
            'deposit_low' => 'Low deposit',
            'student_pending' => 'Registration pending',
        ];

        return $counts->map(function ($count, $type) use ($labels) {
            return [
                'type' => $type,
                'label' => $labels[$type] ?? ucfirst(str_replace('_', ' ', $type)),
                'count' => (int) $count,
            ];
        })->values()->filter(fn ($item) => $item['count'] > 0)->sortByDesc('count')->values();
    }

    /**
     * Get notifications for dropdown (latest 7 unread + some read) - kept for backward compatibility
     */
    public function getDropdownNotifications(string $filter = 'all'): Collection
    {
        $query = AdminNotification::query()->orderBy('created_at', 'desc');

        if ($filter === 'unread') {
            $query->unread();
        } elseif ($filter === 'critical') {
            $query->where('priority', 'critical');
        }

        return $query->limit(10)->get();
    }

    /**
     * Get grouped notifications for full page
     */
    public function getGroupedNotifications(string $filter = 'all'): array
    {
        $query = AdminNotification::query()->orderBy('created_at', 'desc');

        if ($filter === 'unread') {
            $query->unread();
        } elseif ($filter === 'critical') {
            $query->where('priority', 'critical');
        }

        $notifications = $query->get();

        return [
            'today' => $notifications->filter(fn($n) => $n->created_at->isToday()),
            'yesterday' => $notifications->filter(fn($n) => $n->created_at->isYesterday()),
            'older' => $notifications->filter(fn($n) => $n->created_at < Carbon::yesterday()),
        ];
    }

    /**
     * Mark notification as read
     */
    public function markAsRead(int $id): void
    {
        $notification = AdminNotification::find($id);
        if ($notification) {
            $notification->markAsRead();
        }
    }

    /**
     * Mark all as read
     */
    public function markAllAsRead(): void
    {
        AdminNotification::unread()->update([
            'is_read' => true,
            'read_at' => now(),
        ]);
    }

    // Sync methods
    protected function syncExpiringAgreements(): void
    {
        $today = Carbon::today();
        $thirtyDaysFromNow = Carbon::today()->addDays(30);

        $agreements = AgreementHistory::query()
            ->with('student')
            ->whereBetween('to_date', [$today, $thirtyDaysFromNow])
            ->whereHas('student', fn($q) => $q->where('status', Student::STATUS_ACTIVE))
            ->get();

        foreach ($agreements as $agreement) {
            $daysRemaining = Carbon::today()->diffInDays($agreement->to_date, false);
            $priority = $daysRemaining <= 7 ? 'critical' : ($daysRemaining <= 15 ? 'warning' : 'info');

            AdminNotification::updateOrCreate(
                [
                    'type' => 'agreement_expiring',
                    'related_id' => $agreement->id,
                    'related_type' => 'AgreementHistory',
                ],
                [
                    'priority' => $priority,
                    'title' => $agreement->student->full_name ?? 'Unknown',
                    'message' => "Agreement expires in {$daysRemaining} days",
                    'meta' => [
                        'student_id' => $agreement->student_id,
                        'room_bed' => $agreement->student->room_bed_identifier ?? 'N/A',
                        'days_remaining' => $daysRemaining,
                        'to_date' => $agreement->to_date->format('d M Y'),
                    ],
                    'action_url' => route('admin.students.show', $agreement->student_id),
                ]
            );
        }
    }

    protected function syncOverdueRents(): void
    {
        $today = Carbon::today();

        $rents = Rent::query()
            ->with('student')
            ->whereNull('paid_at')
            ->where('due_date', '<', $today)
            ->whereHas('student', fn($q) => $q->where('status', Student::STATUS_ACTIVE))
            ->get();

        foreach ($rents as $rent) {
            $daysOverdue = Carbon::parse($rent->due_date)->diffInDays(Carbon::today());

            AdminNotification::updateOrCreate(
                [
                    'type' => 'rent_overdue',
                    'related_id' => $rent->id,
                    'related_type' => 'Rent',
                ],
                [
                    'priority' => 'critical',
                    'title' => $rent->student->full_name ?? 'Unknown',
                    'message' => "Rent overdue by {$daysOverdue} days",
                    'meta' => [
                        'student_id' => $rent->student_id,
                        'room_bed' => $rent->student->room_bed_identifier ?? 'N/A',
                        'days_overdue' => $daysOverdue,
                        'amount' => $rent->amount,
                        'due_date' => $rent->due_date->format('d M Y'),
                    ],
                    'action_url' => route('admin.rents.index'),
                ]
            );
        }
    }

    protected function syncDueRents(): void
    {
        $today = Carbon::today();
        $sevenDaysFromNow = Carbon::today()->addDays(7);

        $rents = Rent::query()
            ->with('student')
            ->whereNull('paid_at')
            ->whereBetween('due_date', [$today, $sevenDaysFromNow])
            ->whereHas('student', fn($q) => $q->where('status', Student::STATUS_ACTIVE))
            ->get();

        foreach ($rents as $rent) {
            $daysRemaining = Carbon::today()->diffInDays($rent->due_date, false);
            $priority = $daysRemaining <= 2 ? 'critical' : 'warning';

            AdminNotification::updateOrCreate(
                [
                    'type' => 'rent_due',
                    'related_id' => $rent->id,
                    'related_type' => 'Rent',
                ],
                [
                    'priority' => $priority,
                    'title' => $rent->student->full_name ?? 'Unknown',
                    'message' => "Rent due in {$daysRemaining} days",
                    'meta' => [
                        'student_id' => $rent->student_id,
                        'room_bed' => $rent->student->room_bed_identifier ?? 'N/A',
                        'days_remaining' => $daysRemaining,
                        'amount' => $rent->amount,
                        'due_date' => $rent->due_date->format('d M Y'),
                    ],
                    'action_url' => route('admin.rents.index'),
                ]
            );
        }
    }

    protected function syncLowDeposits(): void
    {
        $deposits = Deposit::query()
            ->with('student')
            ->where('balance_amount', '<', 0)
            ->whereHas('student', fn($q) => $q->where('status', Student::STATUS_ACTIVE))
            ->get();

        foreach ($deposits as $deposit) {
            $balanceAmount = abs($deposit->balance_amount);
            $priority = $balanceAmount > 5000 ? 'critical' : 'warning';

            AdminNotification::updateOrCreate(
                [
                    'type' => 'deposit_low',
                    'related_id' => $deposit->id,
                    'related_type' => 'Deposit',
                ],
                [
                    'priority' => $priority,
                    'title' => $deposit->student->full_name ?? 'Unknown',
                    'message' => "Negative deposit balance",
                    'meta' => [
                        'student_id' => $deposit->student_id,
                        'room_bed' => $deposit->student->room_bed_identifier ?? 'N/A',
                        'balance' => -$balanceAmount,
                    ],
                    'action_url' => route('admin.deposits.add'),
                ]
            );
        }
    }

    protected function syncPendingStudents(): void
    {
        $students = Student::query()
            ->whereIn('form_status', [
                Student::FORM_STATUS_PENDING,
                Student::FORM_STATUS_DOCUMENTS_UPLOADED,
                Student::FORM_STATUS_BED_ALLOTTED,
                Student::FORM_STATUS_DEPOSIT_PAID,
            ])
            ->where('created_at', '<', Carbon::now()->subDays(2))
            ->get();

        foreach ($students as $student) {
            $daysWaiting = Carbon::parse($student->created_at)->diffInDays(Carbon::now());
            $priority = $daysWaiting > 7 ? 'critical' : 'warning';

            AdminNotification::updateOrCreate(
                [
                    'type' => 'student_pending',
                    'related_id' => $student->id,
                    'related_type' => 'Student',
                ],
                [
                    'priority' => $priority,
                    'title' => $student->full_name ?? 'Unknown',
                    'message' => "Registration pending",
                    'meta' => [
                        'student_id' => $student->id,
                        'days_waiting' => $daysWaiting,
                        'form_status' => $student->form_status,
                    ],
                    'action_url' => route('admin.students.index', ['filter' => 'information_pending']),
                ]
            );
        }
    }
}
