<?php

namespace App\Services;

use App\Models\OtpVerification;
use App\Models\User;
use App\Mail\OtpVerificationMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class OtpService
{
    public function generate(User $user, string $guard): OtpVerification
    {
        $this->invalidatePendingForUser($user->id, $guard);

        $isLocal = app()->environment('local');

        if ($isLocal) {
            $otpCode = config('otp.static_otp', '123456');
            if (config('app.debug')) {
                Log::debug('OTP (local): ' . $otpCode . ' for user ' . $user->id);
            }
        } else {
            $length = (int) config('otp.length', 6);
            $min = (int) str_pad('1', $length, '0', STR_PAD_RIGHT); // 1, 10, 100, ...
            $max = (int) str_repeat('9', $length); // 9, 99, 999, ...
            $otpCode = (string) random_int($min, $max);
        }

        $expiresAt = Carbon::now()->addMinutes(config('otp.expiry_minutes', 5));

        $otp = OtpVerification::create([
            'user_id' => $user->id,
            'guard' => $guard,
            'otp_code' => $otpCode,
            'expires_at' => $expiresAt,
        ]);

        if (!$isLocal) {
            Mail::to($user->email)->send(new OtpVerificationMail($user, $otpCode));
        }

        return $otp;
    }

    public function verify(User $user, string $guard, string $code): bool
    {
        $otp = OtpVerification::query()
            ->where('user_id', $user->id)
            ->where('guard', $guard)
            ->whereNull('used_at')
            ->where('expires_at', '>', now())
            ->latest()
            ->first();

        if (!$otp || !hash_equals($otp->otp_code, $code)) {
            return false;
        }

        $otp->update(['used_at' => now()]);
        return true;
    }

    public function invalidatePendingForUser(int $userId, string $guard): void
    {
        OtpVerification::query()
            ->where('user_id', $userId)
            ->where('guard', $guard)
            ->whereNull('used_at')
            ->update(['used_at' => now()]);
    }

    /**
     * Mask email for UI display: ym****@empoweredmargins.com
     */
    public static function maskEmail(string $email): string
    {
        $parts = explode('@', $email);
        if (count($parts) !== 2) {
            return $email;
        }
        $local = $parts[0];
        $domain = $parts[1];
        $len = strlen($local);
        if ($len <= 2) {
            $masked = str_repeat('*', $len) . '@' . $domain;
        } else {
            $masked = substr($local, 0, 2) . str_repeat('*', min($len - 2, 4)) . '@' . $domain;
        }
        return $masked;
    }
}
