<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Menu;
use App\Models\User;
use App\Models\CustomPageModel;
use App\Helpers\MenuHelper;
use Illuminate\Foundation\Testing\RefreshDatabase;

class MenuManagementTest extends TestCase
{
    use RefreshDatabase;

    protected $superAdmin;
    protected $admin;
    protected $instructor;
    protected $user;

    protected function setUp(): void
    {
        parent::setUp();

        // Create test users with different roles
        $this->superAdmin = User::factory()->create(['role' => 'super_admin']);
        $this->admin = User::factory()->create(['role' => 'admin']);
        $this->instructor = User::factory()->create(['role' => 'instructor']);
        $this->user = User::factory()->create(['role' => 'user']);
    }

    /** @test */
    public function it_can_create_a_menu()
    {
        $menu = Menu::create([
            'title' => 'dashboard',
            'title_text' => 'Dashboard',
            'link_type' => 'custom_page',
            'custom_page' => 'dashboard',
            'order' => 1,
            'is_active' => true,
        ]);

        $this->assertDatabaseHas('menus', [
            'title' => 'dashboard',
            'title_text' => 'Dashboard',
            'link_type' => 'custom_page',
        ]);
    }

    /** @test */
    public function it_can_create_menu_with_submenus()
    {
        $menu = Menu::create([
            'title' => 'settings',
            'title_text' => 'Settings',
            'link_type' => 'placeholder',
            'order' => 1,
            'is_active' => true,
        ]);

        $menu->submenus_array = [
            [
                'title' => 'general',
                'title_text' => 'General Settings',
                'link_type' => 'custom_page',
                'custom_page' => 'general_settings',
                'submenus' => [],
            ],
            [
                'title' => 'api',
                'title_text' => 'API Settings',
                'link_type' => 'custom_page',
                'custom_page' => 'api_configuration',
                'submenus' => [],
            ],
        ];
        $menu->save();

        $this->assertTrue($menu->hasSubmenus());
        $this->assertEquals(2, $menu->countSubmenus());
        $this->assertEquals(1, $menu->getDepth());
    }

    /** @test */
    public function it_can_create_nested_submenus()
    {
        $menu = Menu::create([
            'title' => 'admin',
            'title_text' => 'Administration',
            'link_type' => 'placeholder',
            'order' => 1,
            'is_active' => true,
        ]);

        $menu->submenus_array = [
            [
                'title' => 'users',
                'title_text' => 'Users',
                'link_type' => 'custom_page',
                'custom_page' => 'users',
                'submenus' => [
                    [
                        'title' => 'create_user',
                        'title_text' => 'Create User',
                        'link_type' => 'custom_page',
                        'custom_page' => 'create_user',
                        'submenus' => [
                            [
                                'title' => 'user_form',
                                'title_text' => 'User Form',
                                'link_type' => 'custom_page',
                                'custom_page' => 'user_form',
                                'submenus' => [],
                            ],
                        ],
                    ],
                ],
            ],
        ];
        $menu->save();

        $this->assertTrue($menu->hasSubmenus());
        $this->assertEquals(3, $menu->countSubmenus()); // users, create_user, user_form
        $this->assertEquals(3, $menu->getDepth());
    }

    /** @test */
    public function it_can_query_active_menus()
    {
        Menu::create([
            'title' => 'active_menu',
            'title_text' => 'Active Menu',
            'link_type' => 'custom_page',
            'custom_page' => 'active',
            'order' => 1,
            'is_active' => true,
        ]);

        Menu::create([
            'title' => 'inactive_menu',
            'title_text' => 'Inactive Menu',
            'link_type' => 'custom_page',
            'custom_page' => 'inactive',
            'order' => 2,
            'is_active' => false,
        ]);

        $activeMenus = Menu::active()->get();
        
        $this->assertCount(1, $activeMenus);
        $this->assertEquals('active_menu', $activeMenus->first()->title);
    }

    /** @test */
    public function it_can_order_menus()
    {
        Menu::create(['title' => 'third', 'title_text' => 'Third', 'link_type' => 'placeholder', 'order' => 3, 'is_active' => true]);
        Menu::create(['title' => 'first', 'title_text' => 'First', 'link_type' => 'placeholder', 'order' => 1, 'is_active' => true]);
        Menu::create(['title' => 'second', 'title_text' => 'Second', 'link_type' => 'placeholder', 'order' => 2, 'is_active' => true]);

        $menus = Menu::ordered()->get();
        
        $this->assertEquals('first', $menus[0]->title);
        $this->assertEquals('second', $menus[1]->title);
        $this->assertEquals('third', $menus[2]->title);
    }

    /** @test */
    public function it_returns_link_attribute()
    {
        $menuWithUrl = Menu::create([
            'title' => 'external',
            'title_text' => 'External',
            'link_type' => 'link_url',
            'link_url' => 'https://example.com',
            'order' => 1,
            'is_active' => true,
        ]);

        $menuWithPage = Menu::create([
            'title' => 'internal',
            'title_text' => 'Internal',
            'link_type' => 'custom_page',
            'custom_page' => 'dashboard',
            'order' => 2,
            'is_active' => true,
        ]);

        $this->assertEquals('https://example.com', $menuWithUrl->link);
        $this->assertEquals('dashboard', $menuWithPage->link);
    }

    /** @test */
    public function super_admin_can_access_all_menus()
    {
        $this->actingAs($this->superAdmin);

        $menuItem = ['custom_page' => 'users', 'link_type' => 'custom_page'];
        
        $this->assertTrue(MenuHelper::canAccessMenuItem($menuItem));
    }

    /** @test */
    public function instructor_can_access_allowlisted_items()
    {
        $this->actingAs($this->instructor);

        $allowedItem = ['custom_page' => 'courses', 'link_type' => 'custom_page'];
        $deniedItem = ['custom_page' => 'users', 'link_type' => 'custom_page'];
        
        $this->assertTrue(MenuHelper::canAccessMenuItem($allowedItem));
        $this->assertFalse(MenuHelper::canAccessMenuItem($deniedItem));
    }

    /** @test */
    public function placeholder_items_are_visible_to_authenticated_users()
    {
        $this->actingAs($this->user);

        $placeholderItem = ['link_type' => 'placeholder', 'title_text' => 'Settings'];
        
        $this->assertTrue(MenuHelper::canAccessMenuItem($placeholderItem));
    }

    /** @test */
    public function external_links_are_visible_to_authenticated_users()
    {
        $this->actingAs($this->user);

        $linkItem = ['link_type' => 'link_url', 'link_url' => 'https://example.com'];
        
        $this->assertTrue(MenuHelper::canAccessMenuItem($linkItem));
    }

    /** @test */
    public function it_filters_menu_by_permissions()
    {
        $this->actingAs($this->instructor);

        $menus = [
            [
                'title' => 'courses',
                'title_text' => 'Courses',
                'link_type' => 'custom_page',
                'custom_page' => 'courses',
                'submenus' => [],
            ],
            [
                'title' => 'users',
                'title_text' => 'Users',
                'link_type' => 'custom_page',
                'custom_page' => 'users',
                'submenus' => [],
            ],
        ];

        $filteredMenus = MenuHelper::filterMenuByPermissions($menus);
        
        $this->assertCount(1, $filteredMenus);
        $this->assertEquals('courses', $filteredMenus[0]['title']);
    }

    /** @test */
    public function it_filters_nested_submenus()
    {
        $this->actingAs($this->instructor);

        $menus = [
            [
                'title' => 'academy',
                'title_text' => 'Academy',
                'link_type' => 'placeholder',
                'submenus' => [
                    [
                        'title' => 'courses',
                        'title_text' => 'Courses',
                        'link_type' => 'custom_page',
                        'custom_page' => 'courses',
                        'submenus' => [],
                    ],
                    [
                        'title' => 'users',
                        'title_text' => 'Users',
                        'link_type' => 'custom_page',
                        'custom_page' => 'users',
                        'submenus' => [],
                    ],
                ],
            ],
        ];

        $filteredMenus = MenuHelper::filterMenuByPermissions($menus);
        
        $this->assertCount(1, $filteredMenus);
        $this->assertCount(1, $filteredMenus[0]['submenus']);
        $this->assertEquals('courses', $filteredMenus[0]['submenus'][0]['title']);
    }

    /** @test */
    public function it_removes_parent_without_accessible_children()
    {
        $this->actingAs($this->instructor);

        $menus = [
            [
                'title' => 'admin',
                'title_text' => 'Administration',
                'link_type' => 'placeholder',
                'submenus' => [
                    [
                        'title' => 'users',
                        'title_text' => 'Users',
                        'link_type' => 'custom_page',
                        'custom_page' => 'users',
                        'submenus' => [],
                    ],
                    [
                        'title' => 'settings',
                        'title_text' => 'Settings',
                        'link_type' => 'custom_page',
                        'custom_page' => 'api_configuration',
                        'submenus' => [],
                    ],
                ],
            ],
        ];

        $filteredMenus = MenuHelper::filterMenuByPermissions($menus);
        
        // Should be empty because instructor cannot access any of the children
        $this->assertCount(0, $filteredMenus);
    }

    /** @test */
    public function it_can_add_custom_permission_mapping()
    {
        MenuHelper::addPermissionMapping('blog', 'manage_blog');
        
        $permissionMap = MenuHelper::getPermissionMap();
        
        $this->assertArrayHasKey('blog', $permissionMap);
        $this->assertEquals('manage_blog', $permissionMap['blog']);
    }

    /** @test */
    public function it_can_add_to_instructor_allowlist()
    {
        MenuHelper::addToInstructorAllowlist('my_custom_page');
        
        $allowlist = MenuHelper::getInstructorAllowlist();
        
        $this->assertContains('my_custom_page', $allowlist);
    }

    /** @test */
    public function it_creates_custom_pages()
    {
        $customPage = CustomPageModel::create([
            'page_id' => 'about',
            'page_slug' => 'about-us',
            'status' => 1,
        ]);

        $this->assertDatabaseHas('custom_pages', [
            'page_id' => 'about',
            'page_slug' => 'about-us',
        ]);
    }

    /** @test */
    public function it_can_query_active_custom_pages()
    {
        CustomPageModel::create(['page_id' => 'active', 'page_slug' => 'active-page', 'status' => 1]);
        CustomPageModel::create(['page_id' => 'inactive', 'page_slug' => 'inactive-page', 'status' => 0]);

        $activePages = CustomPageModel::active()->get();
        
        $this->assertCount(1, $activePages);
        $this->assertEquals('active-page', $activePages->first()->page_slug);
    }
}
